// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

package org.emulator.calculator;

import android.graphics.Bitmap;
import android.util.Log;

/**
 * Based on the free HP82240B Printer Simulator by Christoph Giesselink
 */
public class PrinterSimulator {

    private static final String TAG = "PrinterSimulator";
    private boolean debug = false;
    private StringBuilder m_Text = new StringBuilder();
    private StringBuilder textUpdate = new StringBuilder();


    private final int ESC = 27;					// ESC

    private final byte PM_HP82240A = 0;
    private final byte PM_HP82240B = 1;
    private final byte PM_GenericSerial = 2;

    private byte m_ePrinterModel;               // selected printer model
    private boolean m_bExpChar;                 // printing expanded characters
    private boolean m_bUnderLined;              // printing underlined characters
    private boolean m_bEcma94;                  // Roman 8 / ECMA 94 character set

    private boolean m_bEsc;                     // not ESC sequence
    private int m_byGraphLength;                // remaining no. of graphic bytes

    public PrinterSimulator() {
        m_ePrinterModel = PM_HP82240B;			// HP82240B printer

        int maxBitmapHeight = 2048;
        try {
            maxBitmapHeight = Utils.getMaximumTextureSize();
        } catch(Exception ex) {
            Log.d(TAG, "Cannot get the MaximumTextureSize (Set default to 2048). Error: " + ex.getMessage());
        }
        if(maxBitmapHeight == 0)
            maxBitmapHeight = 1024;

        maxBitmapHeight = Math.min(maxBitmapHeight, 8192); //32768);
        MAXPRTLINES = maxBitmapHeight / LINE_HEIGHT;
        mBitmap = Bitmap.createBitmap(LINE_WIDTH, MAXPRTLINES*LINE_HEIGHT, Bitmap.Config.ARGB_8888); //ARGB_8888); //ALPHA_8);
        mBitmap.eraseColor(0xFFFFFFFF);

        reset();								// reset printer state machine
    }

    /**
     * Set the printer type.
     * @param enable true to set the printer as a model 82240A or 82240B otherwise.
     */
    public void setPrinterModel82240A(boolean enable) {
        m_bPrinter82240A = enable;
    }


    /**
     * Change the paper, so we cleanup everything.
     */
    void changePaper() {
        reset();
        m_Text.setLength(0);
        mBitmap.eraseColor(0xFFFFFFFF);
        m_nCurCol = 0;
        m_nCurRow = 0;
    }

    /**
     * Interface definition for a callback to be invoked when the printer just has print something.
     */
    public interface OnPrinterUpdateListener {
        /**
         * Called when the printer just has print something.
         */
        void onPrinterUpdate(String textAppended);
    }

    private OnPrinterUpdateListener onPrinterUpdateListener;

    /**
     * Register a callback to be invoked when the printer just has print something.
     * @param onPrinterUpdateListener The callback that will run
     */
    void setOnPrinterUpdateListener(OnPrinterUpdateListener onPrinterUpdateListener) {
        this.onPrinterUpdateListener = onPrinterUpdateListener;
    }

    /**
     * Interface definition for a callback to be invoked when the printer is out of paper.
     */
    public interface OnPrinterOutOfPaperListener {
        /**
         * Called when the printer just has print something.
         */
        void onPrinterOutOfPaper(int currentLine, int maxLine, int currentPixelRow, int maxPixelRow);
    }

    private OnPrinterOutOfPaperListener onPrinterOutOfPaperListener;

    /**
     * Register a callback to be invoked when the printer is out of paper.
     * @param onPrinterOutOfPaperListener The callback that will run
     */
    public void setOnPrinterOutOfPaperListener(OnPrinterOutOfPaperListener onPrinterOutOfPaperListener) {
        this.onPrinterOutOfPaperListener = onPrinterOutOfPaperListener;
    }

    /**
     * Reset the printer state machine.
     */
    private void reset() {
        m_bExpChar = false;						// printing normal-width characters
        m_bUnderLined = false;					// printing non underlined characters

        // Roman 8 character set for HP82240A and HP82240B, ECMA 94 for serial printer type
        m_bEcma94 = (m_ePrinterModel == PM_GenericSerial);

        m_bEsc = false;							// not ESC sequence
        m_byGraphLength = 0;					// no remaining graphic bytes

        outOfPaper = false;
    }

    /**
     * Printer self test.
     */
    private void selftest() {
        // self test normally run in an endless loop, that's very hard to implement,
        // so this implementation printing all characters only one time and then
        // go back to the communication interface

        int i,nLF;

        reset();								// reset printer state machine

        nLF = 1;
        write((byte)4);				    		// begin with special LF
        write((byte)95);						// '_' instead if ' ' as first character
        for (i = 33; i <= 256; ++i) {			// all ROMAN8 characters
            if (nLF == 0) write((byte)4);		// LF after 24 characters printing
            write((byte)i);						// print character
            nLF = ++nLF % 24;
        }

        // post code, identification number: HP82240A = D, HP82240B = G
        write((byte)(m_ePrinterModel == PM_HP82240A ? 'D' : 'G'));

        // rest of post code is battery state full
        String szPostPrt = "\u0004\u0004BAT: 5\u0004\u0004";

        for (int ci = 0; ci < szPostPrt.length(); ci++) { 	// write post print
            char c = szPostPrt.charAt(ci);
            write((byte)c);
        }
    }

    /**
     * Entry point of the data coming in the printer.
     * @param byData The characters to write to the printer.
     */
    public synchronized void write(int byData) {

        textUpdate.setLength(0);

        do {
            // check for begin of ESC sequence
            if (byData == ESC && !m_bEsc && m_byGraphLength == 0) {
                m_bEsc = true;					// ESC sequence mode
                break;
            }

            if (m_bEsc) {						// byte is ESC command
                switch (byData) {
                    case 255: // reset
                        reset();
                        break;
                    case 254: // start self test
                        selftest();
                        break;
                    case 253: // start printing expanded characters
                        m_bExpChar = true;
                        break;
                    case 252: // return to normal-width characters
                        m_bExpChar = false;
                        break;
                    case 251: // start printing underlined characters
                        m_bUnderLined = true;
                        break;
                    case 250: // stop underlining
                        m_bUnderLined = false;
                        break;
                    case 249: // ECMA 94 character set
                        if (m_ePrinterModel == PM_HP82240B)
                            m_bEcma94 = true;
                        break;
                    case 248: // Roman 8 character set
                        if (m_ePrinterModel == PM_HP82240B)
                            m_bEcma94 = false;
                        break;
                    default:
                        // graphic data
                        if (byData >= 1 && byData <= 166) {
                            // remaining graphic bytes
                            m_byGraphLength = byData;
                        }
                }
                m_bEsc = false;
                break;
            }

            // normal character
            if (m_byGraphLength == 0) { 	    // not a graphic character
                addTextData(byData);            // output to text window
            }

            // output to graphic window
            addGraphData(byData, m_byGraphLength > 0);

            if (m_byGraphLength > 0) {  		// in graphic mode
                --m_byGraphLength;				// graphic character printed
            }
        } while (false);

        if(this.onPrinterUpdateListener != null)
            this.onPrinterUpdateListener.onPrinterUpdate(textUpdate.toString());
    }

    // Text Printer

    /**
     * ROMAN8 Unicode table
     */
    private static final int[] wcRoman8 = new int[]
    {
   		0x2591,
        0x00A0, 0x00F7, 0x00D7, 0x221A, 0x222B, 0x03A3, 0x25B6, 0x03C0,
        0x2202, 0x2264, 0x2265, 0x2260, 0x03B1, 0x2192, 0x2190, 0x00B5,
        0x240A, 0x00B0, 0x00AB, 0x00BB, 0x22A6, 0x2081, 0x2082, 0x00B2,
        0x00B3, 0x1D62, 0x2C7C, 0x2025, 0x2071, 0x02B2, 0x1D4F, 0x207F,
        0x2221, 0x00C0, 0x00C2, 0x00C8, 0x00CA, 0x00CB, 0x00CE, 0x00CF,
        0x00B4, 0x0060, 0x02C6, 0x00A8, 0x02DC, 0x00D9, 0x00DB, 0x20A4,
        0x00AF, 0x00DD, 0x00FD, 0x00B0, 0x00C7, 0x00E7, 0x00D1, 0x00F1,
        0x00A1, 0x00BF, 0x00A4, 0x00A3, 0x00A5, 0x00A7, 0x0192, 0x00A2,
        0x00E2, 0x00EA, 0x00F4, 0x00FB, 0x00E1, 0x00E9, 0x00F3, 0x00FA,
        0x00E0, 0x00E8, 0x00F2, 0x00F9, 0x00E4, 0x00EB, 0x00F6, 0x00FC,
        0x00C5, 0x00EE, 0x00D8, 0x00C6, 0x00E5, 0x00ED, 0x00F8, 0x00E6,
        0x00C4, 0x00EC, 0x00D6, 0x00DC, 0x00C9, 0x00EF, 0x00DF, 0x00D4,
        0x00C1, 0x00C3, 0x00E3, 0x00D0, 0x00F0, 0x00CD, 0x00CC, 0x00D3,
        0x00D2, 0x00D5, 0x00F5, 0x0160, 0x0161, 0x00DA, 0x0178, 0x00FF,
        0x00DE, 0x00FE, 0x00B7, 0x00B5, 0x00B6, 0x00BE, 0x00AD, 0x00BC,
        0x00BD, 0x00AA, 0x00BA, 0x00AB, 0x2587, 0x00BB, 0x00B1, 0x00A0
    };

    /**
     * ECMA94 Unicode table
     */
    private static final int[] wcEcma94 = new int[]
    {
	    0x2591,
	    0x2221, 0x0101, 0x2207, 0x221A, 0x222B, 0x03A3, 0x25B6, 0x03C0,
        0x2202, 0x2264, 0x2265, 0x2260, 0x03B1, 0x2192, 0x2190, 0x2193,
        0x2191, 0x03B3, 0x03B4, 0x03B5, 0x03B7, 0x03B8, 0x03BB, 0x03C1,
        0x03C3, 0x03C4, 0x03C9, 0x0394, 0x03A0, 0x03A9, 0x2587, 0x221E,
        0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
        0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
        0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
        0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
        0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7,
        0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
        0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
        0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
        0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
        0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
        0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
        0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF
    };

    private void addTextData(int byData) {
        do {
            // special LF and LF characters
            if (byData == 0x04 || byData == 0x0A) {
                textUpdate.append('\r');
                textUpdate.append('\n');
                if(debug) Log.d(TAG, "addTextData(" + byData + ")");
                break;
            }

            // unprintable control codes
            if (byData < ' ')
                break;							// no print

            // normal 7bit ASCII character
            //if (byData < 128)
	        if (byData < (256 - wcRoman8.length))
                textUpdate.append((char)byData);
            else {
                //byData -= 128;					// index to table
		        // index to table
		        byData -= (256 - wcRoman8.length);

                if (!m_bEcma94)
                    textUpdate.append((char)wcRoman8[byData]);
                else
                    textUpdate.append((char)wcEcma94[byData]);
            }
        } while (false);

        m_Text.append(textUpdate);
    }

    /**
     * Get all the text sent to the printer.
     * @return All the text.
     */
    public String getText() {
        return m_Text.toString();
    }

    // Graphical Printer

    private final Bitmap mBitmap;

    /**
     * Get the full paper as an image limited by the paper height (see getPaperHeight().
     * The size of the bitmap is limited by the hardware constraint (max texture size).
     * @return The image containing all the paper.
     */
    public Bitmap getImage() {
        return mBitmap;
    }

    /**
     * Get the current printer head position to know the paper length.
     * @return The current printer head position in pixel from the start of the paper in the bitmap.
     */
    int getPaperHeight() {
        return m_nCurRow + 1;
    }

    /**
     * Get the printer title following the configuration.
     * @return The printer title.
     */
    public String getTitle() {
        return "HP-82240" + (m_bPrinter82240A ? "A" : "B") + " Printer";
    }

    private int MAXPRTLINES;				    // maximum printable lines (out of paper)
    private final int LINE_WIDTH		= 166;
    private final int LINE_HEIGHT		= 8;
    private boolean   outOfPaper = false;


    private int    m_nCurCol;						// current column in bitmap
    private int    m_nCurRow;						// current row in bitmap

    private boolean   m_bPrinter82240A = false;		// HP82240A ROMAN8 font only


    private void SetColumn(int byData)
    {
        // set column in line
        if (m_nCurCol >= LINE_WIDTH)
        {
            m_nCurCol = 0;
            m_nCurRow += LINE_HEIGHT;	// new line
        }

        // bitmap is monochrome so only modify 1 bit
        for (int i = 0; i < 8; ++i)				// draw each bit
        {
            if ((byData & 0x1) == 0x1)			// bit set?
            {
                //m_pbyPrt[nPos] |= nMask;		// set bit in bitmap
                mBitmap.setPixel(m_nCurCol, m_nCurRow + i, 0x00000000);
            }
            byData >>= 1;						// next data bit
        }

        ++m_nCurCol;
    }

    private void SetSeparatorColumn()
    {
	    if (m_nCurCol > 0)						// not in first column
	    {
		    int byData = m_bUnderLined ? 0x80 : 0x00;

		    // no. of separator columns
		    int i = m_bExpChar ? 2 : 1;

		    // write separator columns when not in last column
		    while (--i >= 0 && m_nCurCol < LINE_WIDTH)
		    {
			    SetColumn(byData);				// set empty column
		    }
	    }
    }

    private void addGraphData(int byData, boolean bGraphicData) {
        if (m_nCurRow >= MAXPRTLINES*LINE_HEIGHT) // reached bitmap size
        {
            if(!outOfPaper && this.onPrinterOutOfPaperListener != null)
                this.onPrinterOutOfPaperListener.onPrinterOutOfPaper(m_nCurRow / LINE_HEIGHT, MAXPRTLINES, m_nCurRow, MAXPRTLINES*LINE_HEIGHT);
            outOfPaper = true;
            return;								// paper empty
        }

        if (bGraphicData)
        {
            SetColumn(byData);
        }
        else
        {
            do
            {
                if (byData == 4 || byData == 10) // LF character
                {
                    m_nCurCol = 0;				// begin on left
                    m_nCurRow += LINE_HEIGHT;	// new line
//                    if (::IsWindow(m_hWnd))
//                    {
//                        ScrollBarAdjust();		// adjust scroll bar to new length
//                        ScrollBarBottom();		// goto bottom of view
//                        InvalidateRect(NULL,FALSE);
//                    }
                    break;
                }
                if (byData >= ' ')				// printable char
                {
                    int[] sChar;
                    int     i;

                    byData -= ' ';				// space is first printable character

                    if (m_bPrinter82240A)		// HP82240A Roman 8 character set
                    {
                        sChar = sFontRoman8_A[byData];
                    }
                    else						// HP82240B Roman 8 or ECMA 94 character set
                    {
                        sChar = !m_bEcma94
						  ? sFontRoman8_B[byData]
						  : sFontEcma94_B[byData];
                    }

                    SetSeparatorColumn();

                    for (i = 0; i < 5; ++i)
                    {
                        byData = sChar[i];
                        if (m_bUnderLined)	    // printing underlined characters
                        {
                            byData |= 0x80;		// set underline bit
                        }

                        SetColumn(byData);		// set column
                        if (m_bExpChar)		    // printing expanded characters
                        {
                            SetColumn(byData);	// set column twice
                        }
                    }

                    SetSeparatorColumn();
                }
            }
            while (false);
        }
    }

    // the printer font data

    /**
     * HP82240A ROMAN8 font table.
     */
    private static final int[][] sFontRoman8_A =
            {
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 32
                    {0x00, 0x00, 0x5F, 0x00, 0x00},        // 33
                    {0x00, 0x07, 0x00, 0x07, 0x00},        // 34
                    {0x14, 0x7F, 0x14, 0x7F, 0x14},        // 35
                    {0x24, 0x2A, 0x7F, 0x2A, 0x12},        // 36
                    {0x23, 0x13, 0x08, 0x64, 0x62},        // 37
                    {0x36, 0x49, 0x56, 0x20, 0x50},        // 38
                    {0x00, 0x00, 0x07, 0x00, 0x00},        // 39
                    {0x00, 0x1C, 0x22, 0x41, 0x00},        // 40
                    {0x00, 0x41, 0x22, 0x1C, 0x00},        // 41
                    {0x08, 0x2A, 0x1C, 0x2A, 0x08},        // 42
                    {0x08, 0x08, 0x3E, 0x08, 0x08},        // 43
                    {0x00, 0xB0, 0x70, 0x00, 0x00},        // 44
                    {0x08, 0x08, 0x08, 0x08, 0x08},        // 45
                    {0x00, 0x60, 0x60, 0x00, 0x00},        // 46
                    {0x20, 0x10, 0x08, 0x04, 0x02},        // 47
                    {0x3E, 0x51, 0x49, 0x45, 0x3E},        // 48
                    {0x00, 0x42, 0x7F, 0x40, 0x00},        // 49
                    {0x62, 0x51, 0x49, 0x49, 0x46},        // 50
                    {0x22, 0x41, 0x49, 0x49, 0x36},        // 51
                    {0x18, 0x14, 0x12, 0x7F, 0x10},        // 52
                    {0x27, 0x45, 0x45, 0x45, 0x39},        // 53
                    {0x3C, 0x4A, 0x49, 0x49, 0x30},        // 54
                    {0x01, 0x71, 0x09, 0x05, 0x03},        // 55
                    {0x36, 0x49, 0x49, 0x49, 0x36},        // 56
                    {0x06, 0x49, 0x49, 0x29, 0x1E},        // 57
                    {0x00, 0x36, 0x36, 0x00, 0x00},        // 58
                    {0x00, 0xB6, 0x76, 0x00, 0x00},        // 59
                    {0x08, 0x14, 0x22, 0x41, 0x00},        // 60
                    {0x14, 0x14, 0x14, 0x14, 0x14},        // 61
                    {0x41, 0x22, 0x14, 0x08, 0x00},        // 62
                    {0x02, 0x01, 0x51, 0x09, 0x06},        // 63
                    {0x3E, 0x41, 0x5D, 0x49, 0x4E},        // 64
                    {0x7E, 0x09, 0x09, 0x09, 0x7E},        // 65
                    {0x7F, 0x49, 0x49, 0x49, 0x36},        // 66
                    {0x3E, 0x41, 0x41, 0x41, 0x22},        // 67
                    {0x7F, 0x41, 0x41, 0x22, 0x1C},        // 68
                    {0x7F, 0x49, 0x49, 0x49, 0x41},        // 69
                    {0x7F, 0x09, 0x09, 0x09, 0x01},        // 70
                    {0x3E, 0x41, 0x41, 0x51, 0x72},        // 71
                    {0x7F, 0x08, 0x08, 0x08, 0x7F},        // 72
                    {0x00, 0x41, 0x7F, 0x41, 0x00},        // 73
                    {0x20, 0x40, 0x40, 0x40, 0x3F},        // 74
                    {0x7F, 0x08, 0x14, 0x22, 0x41},        // 75
                    {0x7F, 0x40, 0x40, 0x40, 0x40},        // 76
                    {0x7F, 0x02, 0x0C, 0x02, 0x7F},        // 77
                    {0x7F, 0x04, 0x08, 0x10, 0x7F},        // 78
                    {0x3E, 0x41, 0x41, 0x41, 0x3E},        // 79
                    {0x7F, 0x09, 0x09, 0x09, 0x06},        // 80
                    {0x3E, 0x41, 0x51, 0x21, 0x5E},        // 81
                    {0x7F, 0x09, 0x19, 0x29, 0x46},        // 82
                    {0x26, 0x49, 0x49, 0x49, 0x32},        // 83
                    {0x01, 0x01, 0x7F, 0x01, 0x01},        // 84
                    {0x3F, 0x40, 0x40, 0x40, 0x3F},        // 85
                    {0x07, 0x18, 0x60, 0x18, 0x07},        // 86
                    {0x7F, 0x20, 0x18, 0x20, 0x7F},        // 87
                    {0x63, 0x14, 0x08, 0x14, 0x63},        // 88
                    {0x03, 0x04, 0x78, 0x04, 0x03},        // 89
                    {0x61, 0x51, 0x49, 0x45, 0x43},        // 90
                    {0x00, 0x7F, 0x41, 0x41, 0x00},        // 91
                    {0x02, 0x04, 0x08, 0x10, 0x20},        // 92
                    {0x00, 0x41, 0x41, 0x7F, 0x00},        // 93
                    {0x04, 0x02, 0x01, 0x02, 0x04},        // 94
                    {0x80, 0x80, 0x80, 0x80, 0x80},        // 95
                    {0x00, 0x03, 0x04, 0x00, 0x00},        // 96
                    {0x20, 0x54, 0x54, 0x54, 0x78},        // 97
                    {0x7F, 0x44, 0x44, 0x44, 0x38},        // 98
                    {0x38, 0x44, 0x44, 0x44, 0x44},        // 99
                    {0x38, 0x44, 0x44, 0x44, 0x7F},        // 100
                    {0x38, 0x54, 0x54, 0x54, 0x18},        // 101
                    {0x08, 0x7E, 0x09, 0x02, 0x00},        // 102
                    {0x18, 0xA4, 0xA4, 0xA4, 0x78},        // 103
                    {0x7F, 0x04, 0x04, 0x04, 0x78},        // 104
                    {0x00, 0x44, 0x7D, 0x40, 0x00},        // 105
                    {0x40, 0x80, 0x84, 0x7D, 0x00},        // 106
                    {0x7F, 0x10, 0x28, 0x44, 0x00},        // 107
                    {0x00, 0x41, 0x7F, 0x40, 0x00},        // 108
                    {0x7C, 0x04, 0x18, 0x04, 0x78},        // 109
                    {0x7C, 0x04, 0x04, 0x04, 0x78},        // 110
                    {0x38, 0x44, 0x44, 0x44, 0x38},        // 111
                    {0xFC, 0x24, 0x24, 0x24, 0x18},        // 112
                    {0x18, 0x24, 0x24, 0x24, 0xFC},        // 113
                    {0x7C, 0x08, 0x04, 0x04, 0x04},        // 114
                    {0x48, 0x54, 0x54, 0x54, 0x24},        // 115
                    {0x04, 0x3F, 0x44, 0x20, 0x00},        // 116
                    {0x3C, 0x40, 0x40, 0x40, 0x7C},        // 117
                    {0x1C, 0x20, 0x40, 0x20, 0x1C},        // 118
                    {0x3C, 0x40, 0x30, 0x40, 0x3C},        // 119
                    {0x44, 0x28, 0x10, 0x28, 0x44},        // 120
                    {0x1C, 0xA0, 0xA0, 0xA0, 0x7C},        // 121
                    {0x44, 0x64, 0x54, 0x4C, 0x44},        // 122
                    {0x08, 0x36, 0x41, 0x41, 0x00},        // 123
                    {0x00, 0x00, 0x7F, 0x00, 0x00},        // 124
                    {0x00, 0x41, 0x41, 0x36, 0x08},        // 125
                    {0x08, 0x04, 0x08, 0x10, 0x08},        // 126
                    {0x55, 0x2A, 0x55, 0x2A, 0x55},        // 127
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 128
                    {0x08, 0x08, 0x2A, 0x08, 0x08},        // 129
                    {0x22, 0x14, 0x08, 0x14, 0x22},        // 130
                    {0x10, 0x20, 0x7F, 0x01, 0x01},        // 131
                    {0x20, 0x40, 0x3E, 0x01, 0x02},        // 132
                    {0x41, 0x63, 0x55, 0x49, 0x63},        // 133
                    {0x7F, 0x7F, 0x3E, 0x1C, 0x08},        // 134
                    {0x04, 0x7C, 0x04, 0x7C, 0x04},        // 135
                    {0x30, 0x48, 0x49, 0x52, 0x3C},        // 136
                    {0x50, 0x58, 0x54, 0x52, 0x51},        // 137
                    {0x51, 0x52, 0x54, 0x58, 0x50},        // 138
                    {0x14, 0x34, 0x1C, 0x16, 0x14},        // 139
                    {0x30, 0x48, 0x48, 0x30, 0x48},        // 140
                    {0x08, 0x08, 0x2A, 0x1C, 0x08},        // 141
                    {0x08, 0x1C, 0x2A, 0x08, 0x08},        // 142
                    {0x80, 0x78, 0x40, 0x40, 0x38},        // 143
                    {0x0F, 0x08, 0x00, 0x78, 0x28},        // 144
                    {0x00, 0x07, 0x05, 0x07, 0x00},        // 145
                    {0x08, 0x14, 0x2A, 0x14, 0x22},        // 146
                    {0x22, 0x14, 0x2A, 0x14, 0x08},        // 147
                    {0x7F, 0x08, 0x08, 0x08, 0x08},        // 148
                    {0x00, 0x00, 0xF8, 0x00, 0x00},        // 149
                    {0x00, 0xE8, 0xA8, 0xB8, 0x00},        // 150
                    {0x00, 0x1D, 0x15, 0x17, 0x00},        // 151
                    {0x00, 0x15, 0x15, 0x1F, 0x00},        // 152
                    {0x00, 0x00, 0x68, 0x80, 0x00},        // 153
                    {0x00, 0x80, 0x80, 0x74, 0x00},        // 154
                    {0x60, 0x60, 0x00, 0x60, 0x60},        // 155
                    {0x00, 0x00, 0x0D, 0x10, 0x00},        // 156
                    {0x00, 0x10, 0x10, 0x0D, 0x00},        // 157
                    {0x00, 0x1F, 0x04, 0x0A, 0x10},        // 158
                    {0x00, 0x1E, 0x02, 0x02, 0x1C},        // 159
                    {0x60, 0x50, 0x58, 0x64, 0x42},        // 160
                    {0x78, 0x14, 0x15, 0x16, 0x78},        // 161
                    {0x78, 0x16, 0x15, 0x16, 0x78},        // 162
                    {0x7C, 0x54, 0x55, 0x56, 0x44},        // 163
                    {0x7C, 0x56, 0x55, 0x56, 0x44},        // 164
                    {0x7C, 0x55, 0x54, 0x55, 0x44},        // 165
                    {0x00, 0x46, 0x7D, 0x46, 0x00},        // 166
                    {0x00, 0x45, 0x7C, 0x45, 0x00},        // 167
                    {0x00, 0x00, 0x02, 0x01, 0x00},        // 168
                    {0x00, 0x01, 0x02, 0x00, 0x00},        // 169
                    {0x00, 0x02, 0x01, 0x02, 0x00},        // 170
                    {0x00, 0x01, 0x00, 0x01, 0x00},        // 171
                    {0x02, 0x01, 0x02, 0x04, 0x02},        // 172
                    {0x3C, 0x41, 0x42, 0x40, 0x3C},        // 173
                    {0x38, 0x42, 0x41, 0x42, 0x38},        // 174
                    {0x58, 0x7E, 0x59, 0x41, 0x02},        // 175
                    {0x01, 0x01, 0x01, 0x01, 0x01},        // 176
                    {0x0C, 0x10, 0x62, 0x11, 0x0C},        // 177
                    {0x18, 0xA0, 0xA2, 0xA1, 0x78},        // 178
                    {0x00, 0x07, 0x05, 0x07, 0x00},        // 179
                    {0x1E, 0xA1, 0xA1, 0x61, 0x12},        // 180
                    {0x18, 0xA4, 0xA4, 0x64, 0x24},        // 181
                    {0x7E, 0x09, 0x12, 0x21, 0x7C},        // 182
                    {0x7A, 0x09, 0x0A, 0x09, 0x70},        // 183
                    {0x00, 0x00, 0x7D, 0x00, 0x00},        // 184
                    {0x30, 0x48, 0x45, 0x40, 0x20},        // 185
                    {0x5D, 0x22, 0x22, 0x22, 0x5D},        // 186
                    {0x48, 0x7E, 0x49, 0x41, 0x02},        // 187
                    {0x2B, 0x2C, 0x78, 0x2C, 0x2B},        // 188
                    {0x08, 0x56, 0x55, 0x35, 0x08},        // 189
                    {0x40, 0x48, 0x3E, 0x09, 0x01},        // 190
                    {0x18, 0x24, 0x7E, 0x24, 0x24},        // 191
                    {0x20, 0x56, 0x55, 0x56, 0x78},        // 192
                    {0x38, 0x56, 0x55, 0x56, 0x48},        // 193
                    {0x30, 0x4A, 0x49, 0x4A, 0x30},        // 194
                    {0x30, 0x42, 0x41, 0x42, 0x30},        // 195
                    {0x20, 0x54, 0x56, 0x55, 0x78},        // 196
                    {0x38, 0x54, 0x56, 0x55, 0x48},        // 197
                    {0x30, 0x48, 0x4A, 0x49, 0x30},        // 198
                    {0x38, 0x40, 0x42, 0x41, 0x38},        // 199
                    {0x20, 0x55, 0x56, 0x54, 0x78},        // 200
                    {0x38, 0x55, 0x56, 0x54, 0x48},        // 201
                    {0x30, 0x49, 0x4A, 0x48, 0x30},        // 202
                    {0x38, 0x41, 0x42, 0x40, 0x38},        // 203
                    {0x20, 0x55, 0x54, 0x55, 0x78},        // 204
                    {0x38, 0x55, 0x54, 0x55, 0x48},        // 205
                    {0x30, 0x49, 0x48, 0x49, 0x30},        // 206
                    {0x38, 0x41, 0x40, 0x41, 0x38},        // 207
                    {0x78, 0x17, 0x15, 0x17, 0x78},        // 208
                    {0x00, 0x4A, 0x79, 0x42, 0x00},        // 209
                    {0x5C, 0x32, 0x2A, 0x26, 0x1D},        // 210
                    {0x7E, 0x09, 0x7E, 0x49, 0x49},        // 211
                    {0x20, 0x57, 0x55, 0x57, 0x78},        // 212
                    {0x00, 0x48, 0x7A, 0x41, 0x00},        // 213
                    {0x48, 0x34, 0x1C, 0x16, 0x09},        // 214
                    {0x24, 0x54, 0x38, 0x54, 0x48},        // 215
                    {0x78, 0x15, 0x14, 0x15, 0x78},        // 216
                    {0x00, 0x49, 0x7A, 0x40, 0x00},        // 217
                    {0x3C, 0x43, 0x42, 0x43, 0x3C},        // 218
                    {0x3C, 0x41, 0x40, 0x41, 0x3C},        // 219
                    {0x7C, 0x54, 0x56, 0x55, 0x44},        // 220
                    {0x00, 0x49, 0x78, 0x41, 0x00},        // 221
                    {0xFE, 0x25, 0x25, 0x25, 0x1A},        // 222
                    {0x38, 0x46, 0x45, 0x46, 0x38},        // 223
                    {0x78, 0x14, 0x16, 0x15, 0x78},        // 224
                    {0x7A, 0x15, 0x16, 0x15, 0x78},        // 225
                    {0x22, 0x55, 0x56, 0x55, 0x78},        // 226
                    {0x08, 0x7F, 0x49, 0x22, 0x1C},        // 227
                    {0x30, 0x48, 0x4A, 0x3F, 0x02},        // 228
                    {0x00, 0x44, 0x7E, 0x45, 0x00},        // 229
                    {0x00, 0x45, 0x7E, 0x44, 0x00},        // 230
                    {0x38, 0x44, 0x46, 0x45, 0x38},        // 231
                    {0x38, 0x45, 0x46, 0x44, 0x38},        // 232
                    {0x3A, 0x45, 0x46, 0x45, 0x38},        // 233
                    {0x30, 0x4A, 0x49, 0x4A, 0x31},        // 234
                    {0x48, 0x55, 0x56, 0x55, 0x24},        // 235
                    {0x40, 0x51, 0x2A, 0x09, 0x00},        // 236
                    {0x3C, 0x40, 0x42, 0x41, 0x3C},        // 237
                    {0x0C, 0x11, 0x60, 0x11, 0x0C},        // 238
                    {0x1C, 0xA1, 0xA0, 0xA1, 0x7C},        // 239
                    {0x41, 0x7F, 0x4B, 0x0A, 0x04},        // 240
                    {0x7F, 0x22, 0x22, 0x22, 0x1C},        // 241
                    {0x00, 0x00, 0x0C, 0x0C, 0x00},        // 242
                    {0x80, 0x78, 0x40, 0x40, 0x38},        // 243
                    {0x06, 0x0F, 0x7F, 0x01, 0x7F},        // 244
                    {0x15, 0x1F, 0x38, 0x24, 0x72},        // 245
                    {0x04, 0x04, 0x04, 0x04, 0x04},        // 246
                    {0x17, 0x08, 0x34, 0x22, 0x70},        // 247
                    {0x17, 0x08, 0x04, 0x6A, 0x58},        // 248
                    {0x00, 0x25, 0x2B, 0x2E, 0x20},        // 249
                    {0x26, 0x29, 0x29, 0x29, 0x26},        // 250
                    {0x08, 0x14, 0x2A, 0x14, 0x22},        // 251
                    {0x7F, 0x7F, 0x7F, 0x7F, 0x7F},        // 252
                    {0x22, 0x14, 0x2A, 0x14, 0x08},        // 253
                    {0x00, 0x24, 0x2E, 0x24, 0x00},        // 254
                    {0x00, 0x00, 0x00, 0x00, 0x00}         // 255
            };

    /**
     * HP82240B ROMAN8 font table.
     */
    private static final int[][] sFontRoman8_B =
            {
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 32
                    {0x00, 0x00, 0x5F, 0x00, 0x00},        // 33
                    {0x00, 0x07, 0x00, 0x07, 0x00},        // 34
                    {0x14, 0x7F, 0x14, 0x7F, 0x14},        // 35
                    {0x24, 0x2A, 0x7F, 0x2A, 0x12},        // 36
                    {0x23, 0x13, 0x08, 0x64, 0x62},        // 37
                    {0x36, 0x49, 0x56, 0x20, 0x50},        // 38
                    {0x00, 0x00, 0x07, 0x00, 0x00},        // 39
                    {0x00, 0x1C, 0x22, 0x41, 0x00},        // 40
                    {0x00, 0x41, 0x22, 0x1C, 0x00},        // 41
                    {0x08, 0x2A, 0x1C, 0x2A, 0x08},        // 42
                    {0x08, 0x08, 0x3E, 0x08, 0x08},        // 43
                    {0x00, 0xB0, 0x70, 0x00, 0x00},        // 44
                    {0x08, 0x08, 0x08, 0x08, 0x08},        // 45
                    {0x00, 0x60, 0x60, 0x00, 0x00},        // 46
                    {0x20, 0x10, 0x08, 0x04, 0x02},        // 47
                    {0x3C, 0x62, 0x52, 0x4A, 0x3C},        // 48
                    {0x00, 0x44, 0x7E, 0x40, 0x00},        // 49
                    {0x64, 0x52, 0x4A, 0x4A, 0x44},        // 50
                    {0x24, 0x42, 0x4A, 0x4A, 0x34},        // 51
                    {0x10, 0x18, 0x14, 0x7E, 0x10},        // 52
                    {0x2E, 0x4A, 0x4A, 0x4A, 0x32},        // 53
                    {0x38, 0x54, 0x52, 0x52, 0x20},        // 54
                    {0x02, 0x62, 0x12, 0x0A, 0x06},        // 55
                    {0x34, 0x4A, 0x4A, 0x4A, 0x34},        // 56
                    {0x04, 0x4A, 0x4A, 0x2A, 0x1C},        // 57
                    {0x00, 0x36, 0x36, 0x00, 0x00},        // 58
                    {0x00, 0xB6, 0x76, 0x00, 0x00},        // 59
                    {0x08, 0x14, 0x22, 0x41, 0x00},        // 60
                    {0x14, 0x14, 0x14, 0x14, 0x14},        // 61
                    {0x41, 0x22, 0x14, 0x08, 0x00},        // 62
                    {0x02, 0x01, 0x51, 0x09, 0x06},        // 63
                    {0x3E, 0x41, 0x5D, 0x49, 0x4E},        // 64
                    {0x7C, 0x12, 0x12, 0x12, 0x7C},        // 65
                    {0x7E, 0x4A, 0x4A, 0x4A, 0x34},        // 66
                    {0x3C, 0x42, 0x42, 0x42, 0x24},        // 67
                    {0x7E, 0x42, 0x42, 0x24, 0x18},        // 68
                    {0x7E, 0x4A, 0x4A, 0x4A, 0x42},        // 69
                    {0x7E, 0x0A, 0x0A, 0x0A, 0x02},        // 70
                    {0x3C, 0x42, 0x42, 0x52, 0x74},        // 71
                    {0x7E, 0x08, 0x08, 0x08, 0x7E},        // 72
                    {0x00, 0x42, 0x7E, 0x42, 0x00},        // 73
                    {0x20, 0x40, 0x40, 0x40, 0x3E},        // 74
                    {0x7E, 0x08, 0x14, 0x22, 0x40},        // 75
                    {0x7E, 0x40, 0x40, 0x40, 0x40},        // 76
                    {0x7E, 0x04, 0x18, 0x04, 0x7E},        // 77
                    {0x7E, 0x04, 0x08, 0x10, 0x7E},        // 78
                    {0x3C, 0x42, 0x42, 0x42, 0x3C},        // 79
                    {0x7E, 0x12, 0x12, 0x12, 0x0C},        // 80
                    {0x3C, 0x42, 0x52, 0x22, 0x5C},        // 81
                    {0x7E, 0x12, 0x12, 0x32, 0x4C},        // 82
                    {0x24, 0x4A, 0x4A, 0x4A, 0x30},        // 83
                    {0x02, 0x02, 0x7E, 0x02, 0x02},        // 84
                    {0x3E, 0x40, 0x40, 0x40, 0x3E},        // 85
                    {0x0E, 0x30, 0x40, 0x30, 0x0E},        // 86
                    {0x7E, 0x20, 0x18, 0x20, 0x7E},        // 87
                    {0x42, 0x24, 0x18, 0x24, 0x42},        // 88
                    {0x06, 0x08, 0x70, 0x08, 0x06},        // 89
                    {0x62, 0x52, 0x4A, 0x46, 0x42},        // 90
                    {0x00, 0x7F, 0x41, 0x41, 0x00},        // 91
                    {0x02, 0x04, 0x08, 0x10, 0x20},        // 92
                    {0x00, 0x41, 0x41, 0x7F, 0x00},        // 93
                    {0x04, 0x02, 0x01, 0x02, 0x04},        // 94
                    {0x80, 0x80, 0x80, 0x80, 0x80},        // 95
                    {0x00, 0x03, 0x04, 0x00, 0x00},        // 96
                    {0x20, 0x54, 0x54, 0x54, 0x78},        // 97
                    {0x7F, 0x44, 0x44, 0x44, 0x38},        // 98
                    {0x38, 0x44, 0x44, 0x44, 0x44},        // 99
                    {0x38, 0x44, 0x44, 0x44, 0x7F},        // 100
                    {0x38, 0x54, 0x54, 0x54, 0x18},        // 101
                    {0x08, 0x7E, 0x09, 0x02, 0x00},        // 102
                    {0x18, 0xA4, 0xA4, 0xA4, 0x78},        // 103
                    {0x7F, 0x04, 0x04, 0x04, 0x78},        // 104
                    {0x00, 0x44, 0x7D, 0x40, 0x00},        // 105
                    {0x40, 0x80, 0x84, 0x7D, 0x00},        // 106
                    {0x7F, 0x10, 0x28, 0x44, 0x00},        // 107
                    {0x00, 0x41, 0x7F, 0x40, 0x00},        // 108
                    {0x7C, 0x04, 0x38, 0x04, 0x78},        // 109
                    {0x7C, 0x04, 0x04, 0x04, 0x78},        // 110
                    {0x38, 0x44, 0x44, 0x44, 0x38},        // 111
                    {0xFC, 0x24, 0x24, 0x24, 0x18},        // 112
                    {0x18, 0x24, 0x24, 0x24, 0xFC},        // 113
                    {0x7C, 0x08, 0x04, 0x04, 0x04},        // 114
                    {0x48, 0x54, 0x54, 0x54, 0x24},        // 115
                    {0x04, 0x3F, 0x44, 0x20, 0x00},        // 116
                    {0x3C, 0x40, 0x40, 0x40, 0x7C},        // 117
                    {0x1C, 0x20, 0x40, 0x20, 0x1C},        // 118
                    {0x3C, 0x40, 0x30, 0x40, 0x3C},        // 119
                    {0x44, 0x28, 0x10, 0x28, 0x44},        // 120
                    {0x1C, 0xA0, 0xA0, 0xA0, 0x7C},        // 121
                    {0x44, 0x64, 0x54, 0x4C, 0x44},        // 122
                    {0x08, 0x36, 0x41, 0x41, 0x00},        // 123
                    {0x00, 0x00, 0x7F, 0x00, 0x00},        // 124
                    {0x00, 0x41, 0x41, 0x36, 0x08},        // 125
                    {0x08, 0x04, 0x08, 0x10, 0x08},        // 126
                    {0x55, 0x2A, 0x55, 0x2A, 0x55},        // 127
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 128
                    {0x08, 0x08, 0x2A, 0x08, 0x08},        // 129
                    {0x22, 0x14, 0x08, 0x14, 0x22},        // 130
                    {0x10, 0x20, 0x7F, 0x01, 0x01},        // 131
                    {0x20, 0x40, 0x3E, 0x01, 0x02},        // 132
                    {0x41, 0x63, 0x55, 0x49, 0x63},        // 133
                    {0x7F, 0x7F, 0x3E, 0x1C, 0x08},        // 134
                    {0x04, 0x7C, 0x04, 0x7C, 0x04},        // 135
                    {0x30, 0x49, 0x4A, 0x4C, 0x38},        // 136
                    {0x50, 0x58, 0x54, 0x52, 0x51},        // 137
                    {0x51, 0x52, 0x54, 0x58, 0x50},        // 138
                    {0x14, 0x34, 0x1C, 0x16, 0x14},        // 139
                    {0x30, 0x48, 0x48, 0x30, 0x48},        // 140
                    {0x08, 0x08, 0x2A, 0x1C, 0x08},        // 141
                    {0x08, 0x1C, 0x2A, 0x08, 0x08},        // 142
                    {0x7C, 0x20, 0x20, 0x1C, 0x20},        // 143
                    {0x0F, 0x08, 0x00, 0x78, 0x28},        // 144
                    {0x00, 0x07, 0x05, 0x07, 0x00},        // 145
                    {0x08, 0x14, 0x2A, 0x14, 0x22},        // 146
                    {0x22, 0x14, 0x2A, 0x14, 0x08},        // 147
                    {0x7F, 0x08, 0x08, 0x08, 0x08},        // 148
                    {0x00, 0x00, 0xF8, 0x00, 0x00},        // 149
                    {0x00, 0xE8, 0xA8, 0xB8, 0x00},        // 150
                    {0x00, 0x1D, 0x15, 0x17, 0x00},        // 151
                    {0x00, 0x15, 0x15, 0x1F, 0x00},        // 152
                    {0x00, 0x00, 0x68, 0x80, 0x00},        // 153
                    {0x00, 0x80, 0x80, 0x74, 0x00},        // 154
                    {0x60, 0x60, 0x00, 0x60, 0x60},        // 155
                    {0x00, 0x00, 0x0D, 0x10, 0x00},        // 156
                    {0x00, 0x10, 0x10, 0x0D, 0x00},        // 157
                    {0x00, 0x1F, 0x04, 0x0A, 0x10},        // 158
                    {0x00, 0x1E, 0x02, 0x02, 0x1C},        // 159
                    {0x60, 0x50, 0x58, 0x64, 0x42},        // 160
                    {0x78, 0x15, 0x16, 0x14, 0x78},        // 161
                    {0x78, 0x16, 0x15, 0x16, 0x78},        // 162
                    {0x7C, 0x55, 0x56, 0x54, 0x44},        // 163
                    {0x7C, 0x56, 0x55, 0x56, 0x44},        // 164
                    {0x7C, 0x55, 0x54, 0x55, 0x44},        // 165
                    {0x00, 0x46, 0x7D, 0x46, 0x00},        // 166
                    {0x00, 0x45, 0x7C, 0x45, 0x00},        // 167
                    {0x00, 0x00, 0x02, 0x01, 0x00},        // 168
                    {0x00, 0x01, 0x02, 0x00, 0x00},        // 169
                    {0x00, 0x02, 0x01, 0x02, 0x00},        // 170
                    {0x00, 0x01, 0x00, 0x01, 0x00},        // 171
                    {0x02, 0x01, 0x02, 0x04, 0x02},        // 172
                    {0x3C, 0x41, 0x42, 0x40, 0x3C},        // 173
                    {0x38, 0x42, 0x41, 0x42, 0x38},        // 174
                    {0x58, 0x7E, 0x59, 0x41, 0x02},        // 175
                    {0x01, 0x01, 0x01, 0x01, 0x01},        // 176
                    {0x04, 0x08, 0x72, 0x09, 0x04},        // 177
                    {0x18, 0xA0, 0xA2, 0xA1, 0x78},        // 178
                    {0x00, 0x07, 0x05, 0x07, 0x00},        // 179
                    {0x1E, 0xA1, 0xA1, 0x61, 0x12},        // 180
                    {0x18, 0xA4, 0xA4, 0x64, 0x24},        // 181
                    {0x7C, 0x0A, 0x11, 0x22, 0x7D},        // 182
                    {0x78, 0x0A, 0x09, 0x0A, 0x71},        // 183
                    {0x00, 0x00, 0x7D, 0x00, 0x00},        // 184
                    {0x30, 0x48, 0x45, 0x40, 0x20},        // 185
                    {0x5D, 0x22, 0x22, 0x22, 0x5D},        // 186
                    {0x48, 0x7E, 0x49, 0x41, 0x02},        // 187
                    {0x2B, 0x2C, 0x78, 0x2C, 0x2B},        // 188
                    {0x08, 0x56, 0x55, 0x35, 0x08},        // 189
                    {0x40, 0x48, 0x3E, 0x09, 0x01},        // 190
                    {0x18, 0x24, 0x7E, 0x24, 0x24},        // 191
                    {0x20, 0x56, 0x55, 0x56, 0x78},        // 192
                    {0x38, 0x56, 0x55, 0x56, 0x18},        // 193
                    {0x30, 0x4A, 0x49, 0x4A, 0x30},        // 194
                    {0x38, 0x42, 0x41, 0x42, 0x78},        // 195
                    {0x20, 0x54, 0x56, 0x55, 0x78},        // 196
                    {0x38, 0x54, 0x56, 0x55, 0x18},        // 197
                    {0x30, 0x48, 0x4A, 0x49, 0x30},        // 198
                    {0x38, 0x40, 0x42, 0x41, 0x78},        // 199
                    {0x20, 0x55, 0x56, 0x54, 0x78},        // 200
                    {0x38, 0x55, 0x56, 0x54, 0x18},        // 201
                    {0x30, 0x49, 0x4A, 0x48, 0x30},        // 202
                    {0x38, 0x41, 0x42, 0x40, 0x78},        // 203
                    {0x20, 0x55, 0x54, 0x55, 0x78},        // 204
                    {0x38, 0x55, 0x54, 0x55, 0x18},        // 205
                    {0x30, 0x49, 0x48, 0x49, 0x30},        // 206
                    {0x38, 0x41, 0x40, 0x41, 0x78},        // 207
                    {0x78, 0x17, 0x15, 0x17, 0x78},        // 208
                    {0x00, 0x4A, 0x79, 0x42, 0x00},        // 209
                    {0x5C, 0x32, 0x2A, 0x26, 0x1D},        // 210
                    {0x7E, 0x09, 0x7E, 0x49, 0x49},        // 211
                    {0x20, 0x57, 0x55, 0x57, 0x78},        // 212
                    {0x00, 0x48, 0x7A, 0x41, 0x00},        // 213
                    {0x58, 0x24, 0x54, 0x48, 0x34},        // 214
                    {0x74, 0x54, 0x7C, 0x54, 0x5C},        // 215
                    {0x78, 0x15, 0x14, 0x15, 0x78},        // 216
                    {0x00, 0x49, 0x7A, 0x40, 0x00},        // 217
                    {0x38, 0x45, 0x44, 0x45, 0x38},        // 218
                    {0x3C, 0x41, 0x40, 0x41, 0x3C},        // 219
                    {0x7C, 0x54, 0x56, 0x55, 0x44},        // 220
                    {0x00, 0x49, 0x78, 0x41, 0x00},        // 221
                    {0xFE, 0x25, 0x25, 0x25, 0x1A},        // 222
                    {0x38, 0x46, 0x45, 0x46, 0x38},        // 223
                    {0x78, 0x14, 0x16, 0x15, 0x78},        // 224
                    {0x7A, 0x15, 0x16, 0x15, 0x78},        // 225
                    {0x22, 0x55, 0x56, 0x55, 0x78},        // 226
                    {0x08, 0x7F, 0x49, 0x22, 0x1C},        // 227
                    {0x30, 0x48, 0x4A, 0x3F, 0x02},        // 228
                    {0x00, 0x44, 0x7E, 0x45, 0x00},        // 229
                    {0x00, 0x45, 0x7E, 0x44, 0x00},        // 230
                    {0x38, 0x44, 0x46, 0x45, 0x38},        // 231
                    {0x38, 0x45, 0x46, 0x44, 0x38},        // 232
                    {0x3A, 0x45, 0x46, 0x45, 0x38},        // 233
                    {0x30, 0x4A, 0x49, 0x4A, 0x31},        // 234
                    {0x48, 0x55, 0x56, 0x55, 0x24},        // 235
                    {0x40, 0x51, 0x2A, 0x09, 0x00},        // 236
                    {0x3C, 0x40, 0x42, 0x41, 0x3C},        // 237
                    {0x04, 0x09, 0x70, 0x09, 0x04},        // 238
                    {0x18, 0xA1, 0xA0, 0xA1, 0x78},        // 239
                    {0x41, 0x7F, 0x55, 0x14, 0x08},        // 240
                    {0x00, 0xFE, 0x24, 0x24, 0x18},        // 241
                    {0x00, 0x18, 0x18, 0x00, 0x00},        // 242
                    {0x7C, 0x20, 0x20, 0x1C, 0x20},        // 243
                    {0x06, 0x4F, 0x7F, 0x01, 0x7F},        // 244
                    {0x15, 0x1F, 0x38, 0x24, 0x72},        // 245
                    {0x04, 0x04, 0x04, 0x04, 0x04},        // 246
                    {0x17, 0x08, 0x34, 0x22, 0x70},        // 247
                    {0x17, 0x08, 0x04, 0x6A, 0x58},        // 248
                    {0x00, 0x28, 0x35, 0x35, 0x2E},        // 249
                    {0x26, 0x29, 0x29, 0x29, 0x26},        // 250
                    {0x08, 0x14, 0x2A, 0x14, 0x22},        // 251
                    {0x7F, 0x7F, 0x7F, 0x7F, 0x7F},        // 252
                    {0x22, 0x14, 0x2A, 0x14, 0x08},        // 253
                    {0x00, 0x24, 0x2E, 0x24, 0x00},        // 254
                    {0x00, 0x00, 0x00, 0x00, 0x00}         // 255
            };

    /**
     * HP82240B ECMA94 font table.
     */
    private static final int[][] sFontEcma94_B =
            {
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 32
                    {0x00, 0x00, 0x5F, 0x00, 0x00},        // 33
                    {0x00, 0x07, 0x00, 0x07, 0x00},        // 34
                    {0x14, 0x7F, 0x14, 0x7F, 0x14},        // 35
                    {0x24, 0x2A, 0x7F, 0x2A, 0x12},        // 36
                    {0x23, 0x13, 0x08, 0x64, 0x62},        // 37
                    {0x36, 0x49, 0x56, 0x20, 0x50},        // 38
                    {0x00, 0x00, 0x07, 0x00, 0x00},        // 39
                    {0x00, 0x1C, 0x22, 0x41, 0x00},        // 40
                    {0x00, 0x41, 0x22, 0x1C, 0x00},        // 41
                    {0x08, 0x2A, 0x1C, 0x2A, 0x08},        // 42
                    {0x08, 0x08, 0x3E, 0x08, 0x08},        // 43
                    {0x00, 0xB0, 0x70, 0x00, 0x00},        // 44
                    {0x08, 0x08, 0x08, 0x08, 0x08},        // 45
                    {0x00, 0x60, 0x60, 0x00, 0x00},        // 46
                    {0x20, 0x10, 0x08, 0x04, 0x02},        // 47
                    {0x3C, 0x62, 0x52, 0x4A, 0x3C},        // 48
                    {0x00, 0x44, 0x7E, 0x40, 0x00},        // 49
                    {0x64, 0x52, 0x4A, 0x4A, 0x44},        // 50
                    {0x24, 0x42, 0x4A, 0x4A, 0x34},        // 51
                    {0x10, 0x18, 0x14, 0x7E, 0x10},        // 52
                    {0x2E, 0x4A, 0x4A, 0x4A, 0x32},        // 53
                    {0x38, 0x54, 0x52, 0x52, 0x20},        // 54
                    {0x02, 0x62, 0x12, 0x0A, 0x06},        // 55
                    {0x34, 0x4A, 0x4A, 0x4A, 0x34},        // 56
                    {0x04, 0x4A, 0x4A, 0x2A, 0x1C},        // 57
                    {0x00, 0x36, 0x36, 0x00, 0x00},        // 58
                    {0x00, 0xB6, 0x76, 0x00, 0x00},        // 59
                    {0x08, 0x14, 0x22, 0x41, 0x00},        // 60
                    {0x14, 0x14, 0x14, 0x14, 0x14},        // 61
                    {0x41, 0x22, 0x14, 0x08, 0x00},        // 62
                    {0x02, 0x01, 0x51, 0x09, 0x06},        // 63
                    {0x3E, 0x41, 0x5D, 0x49, 0x4E},        // 64
                    {0x7C, 0x12, 0x12, 0x12, 0x7C},        // 65
                    {0x7E, 0x4A, 0x4A, 0x4A, 0x34},        // 66
                    {0x3C, 0x42, 0x42, 0x42, 0x24},        // 67
                    {0x7E, 0x42, 0x42, 0x24, 0x18},        // 68
                    {0x7E, 0x4A, 0x4A, 0x4A, 0x42},        // 69
                    {0x7E, 0x0A, 0x0A, 0x0A, 0x02},        // 70
                    {0x3C, 0x42, 0x42, 0x52, 0x74},        // 71
                    {0x7E, 0x08, 0x08, 0x08, 0x7E},        // 72
                    {0x00, 0x42, 0x7E, 0x42, 0x00},        // 73
                    {0x20, 0x40, 0x40, 0x40, 0x3E},        // 74
                    {0x7E, 0x08, 0x14, 0x22, 0x40},        // 75
                    {0x7E, 0x40, 0x40, 0x40, 0x40},        // 76
                    {0x7E, 0x04, 0x18, 0x04, 0x7E},        // 77
                    {0x7E, 0x04, 0x08, 0x10, 0x7E},        // 78
                    {0x3C, 0x42, 0x42, 0x42, 0x3C},        // 79
                    {0x7E, 0x12, 0x12, 0x12, 0x0C},        // 80
                    {0x3C, 0x42, 0x52, 0x22, 0x5C},        // 81
                    {0x7E, 0x12, 0x12, 0x32, 0x4C},        // 82
                    {0x24, 0x4A, 0x4A, 0x4A, 0x30},        // 83
                    {0x02, 0x02, 0x7E, 0x02, 0x02},        // 84
                    {0x3E, 0x40, 0x40, 0x40, 0x3E},        // 85
                    {0x0E, 0x30, 0x40, 0x30, 0x0E},        // 86
                    {0x7E, 0x20, 0x18, 0x20, 0x7E},        // 87
                    {0x42, 0x24, 0x18, 0x24, 0x42},        // 88
                    {0x06, 0x08, 0x70, 0x08, 0x06},        // 89
                    {0x62, 0x52, 0x4A, 0x46, 0x42},        // 90
                    {0x00, 0x7F, 0x41, 0x41, 0x00},        // 91
                    {0x02, 0x04, 0x08, 0x10, 0x20},        // 92
                    {0x00, 0x41, 0x41, 0x7F, 0x00},        // 93
                    {0x04, 0x02, 0x01, 0x02, 0x04},        // 94
                    {0x80, 0x80, 0x80, 0x80, 0x80},        // 95
                    {0x00, 0x03, 0x04, 0x00, 0x00},        // 96
                    {0x20, 0x54, 0x54, 0x54, 0x78},        // 97
                    {0x7F, 0x44, 0x44, 0x44, 0x38},        // 98
                    {0x38, 0x44, 0x44, 0x44, 0x44},        // 99
                    {0x38, 0x44, 0x44, 0x44, 0x7F},        // 100
                    {0x38, 0x54, 0x54, 0x54, 0x18},        // 101
                    {0x08, 0x7E, 0x09, 0x02, 0x00},        // 102
                    {0x18, 0xA4, 0xA4, 0xA4, 0x78},        // 103
                    {0x7F, 0x04, 0x04, 0x04, 0x78},        // 104
                    {0x00, 0x44, 0x7D, 0x40, 0x00},        // 105
                    {0x40, 0x80, 0x84, 0x7D, 0x00},        // 106
                    {0x7F, 0x10, 0x28, 0x44, 0x00},        // 107
                    {0x00, 0x41, 0x7F, 0x40, 0x00},        // 108
                    {0x7C, 0x04, 0x38, 0x04, 0x78},        // 109
                    {0x7C, 0x04, 0x04, 0x04, 0x78},        // 110
                    {0x38, 0x44, 0x44, 0x44, 0x38},        // 111
                    {0xFC, 0x24, 0x24, 0x24, 0x18},        // 112
                    {0x18, 0x24, 0x24, 0x24, 0xFC},        // 113
                    {0x7C, 0x08, 0x04, 0x04, 0x04},        // 114
                    {0x48, 0x54, 0x54, 0x54, 0x24},        // 115
                    {0x04, 0x3F, 0x44, 0x20, 0x00},        // 116
                    {0x3C, 0x40, 0x40, 0x40, 0x7C},        // 117
                    {0x1C, 0x20, 0x40, 0x20, 0x1C},        // 118
                    {0x3C, 0x40, 0x30, 0x40, 0x3C},        // 119
                    {0x44, 0x28, 0x10, 0x28, 0x44},        // 120
                    {0x1C, 0xA0, 0xA0, 0xA0, 0x7C},        // 121
                    {0x44, 0x64, 0x54, 0x4C, 0x44},        // 122
                    {0x08, 0x36, 0x41, 0x41, 0x00},        // 123
                    {0x00, 0x00, 0x7F, 0x00, 0x00},        // 124
                    {0x00, 0x41, 0x41, 0x36, 0x08},        // 125
                    {0x08, 0x04, 0x08, 0x10, 0x08},        // 126
                    {0x55, 0x2A, 0x55, 0x2A, 0x55},        // 127
                    {0x60, 0x50, 0x58, 0x64, 0x42},        // 128
                    {0x45, 0x29, 0x11, 0x29, 0x45},        // 129
                    {0x06, 0x1A, 0x22, 0x1A, 0x06},        // 130
                    {0x10, 0x20, 0x7F, 0x01, 0x01},        // 131
                    {0x20, 0x40, 0x3E, 0x01, 0x02},        // 132
                    {0x41, 0x63, 0x55, 0x49, 0x63},        // 133
                    {0x7F, 0x7F, 0x3E, 0x1C, 0x08},        // 134
                    {0x04, 0x7C, 0x04, 0x7C, 0x04},        // 135
                    {0x30, 0x49, 0x4A, 0x4C, 0x38},        // 136
                    {0x50, 0x58, 0x54, 0x52, 0x51},        // 137
                    {0x51, 0x52, 0x54, 0x58, 0x50},        // 138
                    {0x14, 0x34, 0x1C, 0x16, 0x14},        // 139
                    {0x30, 0x48, 0x48, 0x30, 0x48},        // 140
                    {0x08, 0x08, 0x2A, 0x1C, 0x08},        // 141
                    {0x08, 0x1C, 0x2A, 0x08, 0x08},        // 142
                    {0x10, 0x20, 0x7F, 0x20, 0x10},        // 143
                    {0x04, 0x02, 0x7F, 0x02, 0x04},        // 144
                    {0x08, 0x04, 0x08, 0x70, 0x0C},        // 145
                    {0x30, 0x4A, 0x4D, 0x38, 0x00},        // 146
                    {0x00, 0x38, 0x54, 0x54, 0x54},        // 147
                    {0x08, 0x04, 0x38, 0x04, 0xF8},        // 148
                    {0x3E, 0x49, 0x49, 0x3E, 0x00},        // 149
                    {0x46, 0x28, 0x10, 0x20, 0x40},        // 150
                    {0x80, 0x7C, 0x12, 0x12, 0x0C},        // 151
                    {0x38, 0x44, 0x44, 0x3C, 0x04},        // 152
                    {0x08, 0x04, 0x3C, 0x44, 0x24},        // 153
                    {0x3C, 0x40, 0x30, 0x44, 0x38},        // 154
                    {0x30, 0x28, 0x24, 0x28, 0x30},        // 155
                    {0x01, 0x7F, 0x01, 0x7F, 0x01},        // 156
                    {0x5E, 0x61, 0x01, 0x61, 0x5E},        // 157
                    {0x7F, 0x7F, 0x7F, 0x7F, 0x7F},        // 158
                    {0x18, 0x24, 0x18, 0x24, 0x18},        // 159
                    {0x00, 0x00, 0x00, 0x00, 0x00},        // 160
                    {0x00, 0x00, 0x7D, 0x00, 0x00},        // 161
                    {0x18, 0x24, 0x7E, 0x24, 0x24},        // 162
                    {0x48, 0x7E, 0x49, 0x41, 0x02},        // 163
                    {0x5D, 0x22, 0x22, 0x22, 0x5D},        // 164
                    {0x2B, 0x2C, 0x78, 0x2C, 0x2B},        // 165
                    {0x00, 0x00, 0x77, 0x00, 0x00},        // 166
                    {0x08, 0x56, 0x55, 0x35, 0x08},        // 167
                    {0x00, 0x01, 0x00, 0x01, 0x00},        // 168
                    {0x3E, 0x5D, 0x55, 0x41, 0x3E},        // 169
                    {0x00, 0x28, 0x35, 0x35, 0x2E},        // 170
                    {0x08, 0x14, 0x2A, 0x14, 0x22},        // 171
                    {0x08, 0x08, 0x08, 0x18, 0x00},        // 172
                    {0x08, 0x08, 0x08, 0x08, 0x00},        // 173
                    {0x3E, 0x5D, 0x4D, 0x51, 0x3E},        // 174
                    {0x01, 0x01, 0x01, 0x01, 0x01},        // 175
                    {0x00, 0x07, 0x05, 0x07, 0x00},        // 176
                    {0x00, 0x24, 0x2E, 0x24, 0x00},        // 177
                    {0x00, 0x1D, 0x15, 0x17, 0x00},        // 178
                    {0x00, 0x15, 0x15, 0x1F, 0x00},        // 179
                    {0x00, 0x00, 0x02, 0x01, 0x00},        // 180
                    {0x7C, 0x20, 0x20, 0x1C, 0x20},        // 181
                    {0x06, 0x4F, 0x7F, 0x01, 0x7F},        // 182
                    {0x00, 0x18, 0x18, 0x00, 0x00},        // 183
                    {0x00, 0x80, 0xA0, 0x40, 0x00},        // 184
                    {0x00, 0x09, 0x0F, 0x08, 0x00},        // 185
                    {0x26, 0x29, 0x29, 0x29, 0x26},        // 186
                    {0x22, 0x14, 0x2A, 0x14, 0x08},        // 187
                    {0x17, 0x08, 0x34, 0x22, 0x70},        // 188
                    {0x17, 0x08, 0x04, 0x6A, 0x58},        // 189
                    {0x15, 0x1F, 0x38, 0x24, 0x72},        // 190
                    {0x30, 0x48, 0x45, 0x40, 0x20},        // 191
                    {0x78, 0x15, 0x16, 0x14, 0x78},        // 192
                    {0x78, 0x14, 0x16, 0x15, 0x78},        // 193
                    {0x78, 0x16, 0x15, 0x16, 0x78},        // 194
                    {0x7A, 0x15, 0x16, 0x15, 0x78},        // 195
                    {0x78, 0x15, 0x14, 0x15, 0x78},        // 196
                    {0x78, 0x17, 0x15, 0x17, 0x78},        // 197
                    {0x7E, 0x09, 0x7E, 0x49, 0x49},        // 198
                    {0x1E, 0xA1, 0xA1, 0x61, 0x12},        // 199
                    {0x7C, 0x55, 0x56, 0x54, 0x44},        // 200
                    {0x7C, 0x54, 0x56, 0x55, 0x44},        // 201
                    {0x7C, 0x56, 0x55, 0x56, 0x44},        // 202
                    {0x7C, 0x55, 0x54, 0x55, 0x44},        // 203
                    {0x00, 0x45, 0x7E, 0x44, 0x00},        // 204
                    {0x00, 0x44, 0x7E, 0x45, 0x00},        // 205
                    {0x00, 0x46, 0x7D, 0x46, 0x00},        // 206
                    {0x00, 0x45, 0x7C, 0x45, 0x00},        // 207
                    {0x08, 0x7F, 0x49, 0x22, 0x1C},        // 208
                    {0x7C, 0x0A, 0x11, 0x22, 0x7D},        // 209
                    {0x38, 0x45, 0x46, 0x44, 0x38},        // 210
                    {0x38, 0x44, 0x46, 0x45, 0x38},        // 211
                    {0x38, 0x46, 0x45, 0x46, 0x38},        // 212
                    {0x3A, 0x45, 0x46, 0x45, 0x38},        // 213
                    {0x38, 0x45, 0x44, 0x45, 0x38},        // 214
                    {0x22, 0x14, 0x08, 0x14, 0x22},        // 215
                    {0x5C, 0x32, 0x2A, 0x26, 0x1D},        // 216
                    {0x3C, 0x41, 0x42, 0x40, 0x3C},        // 217
                    {0x3C, 0x40, 0x42, 0x41, 0x3C},        // 218
                    {0x38, 0x42, 0x41, 0x42, 0x38},        // 219
                    {0x3C, 0x41, 0x40, 0x41, 0x3C},        // 220
                    {0x04, 0x08, 0x72, 0x09, 0x04},        // 221
                    {0x41, 0x7F, 0x55, 0x14, 0x08},        // 222
                    {0xFE, 0x25, 0x25, 0x25, 0x1A},        // 223
                    {0x20, 0x55, 0x56, 0x54, 0x78},        // 224
                    {0x20, 0x54, 0x56, 0x55, 0x78},        // 225
                    {0x20, 0x56, 0x55, 0x56, 0x78},        // 226
                    {0x22, 0x55, 0x56, 0x55, 0x78},        // 227
                    {0x20, 0x55, 0x54, 0x55, 0x78},        // 228
                    {0x20, 0x57, 0x55, 0x57, 0x78},        // 229
                    {0x74, 0x54, 0x7C, 0x54, 0x5C},        // 230
                    {0x18, 0xA4, 0xA4, 0x64, 0x24},        // 231
                    {0x38, 0x55, 0x56, 0x54, 0x18},        // 232
                    {0x38, 0x54, 0x56, 0x55, 0x18},        // 233
                    {0x38, 0x56, 0x55, 0x56, 0x18},        // 234
                    {0x38, 0x55, 0x54, 0x55, 0x18},        // 235
                    {0x00, 0x49, 0x7A, 0x40, 0x00},        // 236
                    {0x00, 0x48, 0x7A, 0x41, 0x00},        // 237
                    {0x00, 0x4A, 0x79, 0x42, 0x00},        // 238
                    {0x00, 0x49, 0x78, 0x41, 0x00},        // 239
                    {0x30, 0x48, 0x4A, 0x3F, 0x02},        // 240
                    {0x78, 0x0A, 0x09, 0x0A, 0x71},        // 241
                    {0x30, 0x49, 0x4A, 0x48, 0x30},        // 242
                    {0x30, 0x48, 0x4A, 0x49, 0x30},        // 243
                    {0x30, 0x4A, 0x49, 0x4A, 0x30},        // 244
                    {0x30, 0x4A, 0x49, 0x4A, 0x31},        // 245
                    {0x30, 0x49, 0x48, 0x49, 0x30},        // 246
                    {0x08, 0x08, 0x2A, 0x08, 0x08},        // 247
                    {0x58, 0x24, 0x54, 0x48, 0x34},        // 248
                    {0x38, 0x41, 0x42, 0x40, 0x78},        // 249
                    {0x38, 0x40, 0x42, 0x41, 0x78},        // 250
                    {0x38, 0x42, 0x41, 0x42, 0x78},        // 251
                    {0x38, 0x41, 0x40, 0x41, 0x78},        // 252
                    {0x18, 0xA0, 0xA2, 0xA1, 0x78},        // 253
                    {0x00, 0xFE, 0x24, 0x24, 0x18},        // 254
                    {0x18, 0xA1, 0xA0, 0xA1, 0x78}         // 255
            };
}
