

#define uses_stdlib
#define uses_unixlib
#include "config.h"

#include "input.h"
#include "strlib.h"
#include "misc.h"
#include "error.h"

typedef struct LIFO *LIFO;

struct LIFO {			/* struct used for pushing back characters */
	char	    *buf;	/* what was pushed and will be unpushed */
	char	    *ptr;	/* next character to be unpushed */
        struct LIFO *prev;	/* next string to be unpushed */	
	int	     alloc;	/* nonzero if must Free() this */
};

struct SOURCE {
	char	*filename;
	int	 linenum;
	FILE 	*fp;
	LIFO	 lifo;		/* stack for ungetting lines */
	int	 eof;		/* 1 if reached end-of-file */
	int	 eol;    	/* 1 if reached end-of-line */
};

static	char    *srcmain;

SOURCE	open_source(char *filename)
{
	SOURCE  src = Malloc(sizeof(struct SOURCE));

	srcmain	= Strdup(filename);
	src->fp = fopen(filename, "r");
	if(!src->fp) {
		perror(filename);
		exit(1);
	}
	src->filename = srcmain;
	src->lifo = NULL;
	src->linenum = 0; /* until a char is read, we're on line 0 */
	src->eof = 0;
	src->eol = 1;	  /* bump to line #1 as soon as a char is read */
	return src;
}

static	char nextlifochar(SOURCE src);

char	nextchar(SOURCE src)
{
	if(src->eof) {
		DEBUG(("Attempt to read past EOF\n"));
		return 0;
	}

	if(src->lifo) {
		return nextlifochar(src);
	} else {
		int c;
		c = fgetc(src->fp);
		if(c==EOF) {
			src->eof = 1;
			return 0;
		}
		if(src->eol) {
		/* The \n character belongs to the line it appears on,
		   so the linenum is only incremented when we read the
		   first character of the next line. */
			src->linenum++;
			src->eol = 0;
		}
		switch(c) {
		case EOF:
			src->eof = 1;
			return 0;
		case '\r':
			return nextchar(src);
		case '\n':
			src->eol = 1;
		default:
			return c;
		}
	}
}



char	*nextword(SOURCE src)
{
	char   *buf=Malloc(256);
	int	i, c;

	while( (c=checkchar(src)) && isspace(c) ) {
		(void) nextchar(src);
	}
	i=0;
	while( (c=checkchar(src)) && !isspace(c) ) {
		if(i && i%256==0) {
			buf = Realloc(buf, i+257);
		}
		buf[i++] = nextchar(src);
	}
	buf[i] = 0;
	DEBUG(("returning %s\n",buf));
	return mark(buf);
}




char	*restline(SOURCE src)
{
	const maxline = 255;
	char *buf = Malloc(maxline+1);
	int   i=0, c;
		
	while( (c=nextchar(src)) && c!='\n') {
		if(i==maxline) {
			fatal_error("line too long");
		}
		buf[i++] = c;
	}
	buf[i] = 0;
	return mark(buf);
}

int	getstatus(SOURCE src, char **file, int *line)
{
	if(file) { *file = src ? src->filename : "NONE"; }
	if(line) { *line = src ? src->linenum : 0; }
	return src ? src->eof : 0;
}

void	close_source(SOURCE src)
{
	fclose(src->fp);
	memset(src,-1,sizeof(struct SOURCE));
}

static  void pushback_nodup(char *str, SOURCE src)
{
	LIFO lifo = Malloc(sizeof(struct LIFO));
	lifo->prev = src->lifo;
	lifo->buf  = str;
	lifo->ptr  = lifo->buf;
	lifo->alloc= 0;
	src->lifo = lifo;
}

void	pushback(char *str, SOURCE src)
{
	pushback_nodup(Strdup(str), src);
	src->lifo->alloc = 1;
}

void    include(char *filename, SOURCE src)
{
	char	*text;
	int	 fd, len;

	filename = opquotes(filename);
	fd = open(filename, O_RDONLY);
	if(fd<0) {
		perror(filename);
		fatal_error("Bad INCLUDE directive.");
	}
 	len = file_size(filename);
	text = mmap(NULL, len, PROT_READ, MAP_PRIVATE, fd, 0);

	if(text==NULL || text==(char *)(-1)) {
		perror("mmap");
		abort();
	}
	pushback_nodup(text,src);
}



static	char nextlifochar(SOURCE src)
{
	char c;
	c = *(src->lifo->ptr++);
	if(!*src->lifo->ptr) {
		LIFO prev;
		if(src->lifo->alloc) {
			Free(src->lifo->buf);
		}
		prev = src->lifo->prev;
		Free(src->lifo);
		src->lifo = prev;
	}
	return c;
}




char	checkchar(SOURCE src)
{
	if(src->eof) {
		return 0;
	} else {
		if(src->lifo) {
			return *(src->lifo->ptr);
		} else {
			int c = fgetc(src->fp);
			ungetc(c, src->fp);
			return c;
		}
	}
}



char   *nextquote(SOURCE src, char delim)
{
    int     i, mem;
    char    *s, c;
    char    close_delim = delim;

    while(isspace(checkchar(src))) {
	nextchar(src);
    }
    if( nextchar(src) != delim ) {
        fatal_error(nprintf("delimited string expected", delim));
    }
    mem=0;
    s=Malloc(1);
    i=0; 
    while ( (c = nextchar(src)) && c!=close_delim) {
        if (i>=mem-1) {
            s = Realloc(s, mem+=256);
	}
	s[i++] = c;
    }
    if (!c) {
         fatal_error("runaway delimited string");
    }
    s[i] = 0;
    return mark(s);
}

