/*

Generic string/memory/file handling subroutines.

Copyright (C) 1993 Alex Ramos

Returned-pointer convention: All return values (when pointers)
point to an area that was obtained with Malloc() and marked
for garbage collection with mark(). A garbage_collect() will
call Free() for all pointers that were marked.
Pointers returned by any of these functions must not be passed
to Realloc(), ever.

*/

#define uses_stdlib
#define uses_unixlib
#include "config.h"

#include "strlib.h"
#include "misc.h"

#define TRUE  1
#define FALSE 0



/*

  ERROR HANDLING

*/

static void (*error_fcn)(char *);

void    bind_error_handler(func)
    	   void (*func)();
{
error_fcn = func;
}

void error(mesg)
    char *mesg;
/* Simple error function. Can be bound to an external error handler 
   by use of the bind_error_handler() function. */
{
if(error_fcn)
    (*error_fcn)(mesg);
else
    fprintf(stderr, mesg);
    exit(1);
}





long            long_pow(base, exp)
    long            base, exp;
{
    long            x = 1;
    while (exp--)
        x *= base;
    return x;
}



int             isdecnum(s)
    char           *s;
{
    if (!*s)
        return FALSE;

    while (*s) {
        if (!isdigit(*s))
            return FALSE;
        ++s;
    }

    return TRUE;
}



/*
 *
 * return true if s is a valid hexadecimal integer
 *
 *
 */



int             ishexnum(s)
    char           *s;
{
    if (!*s)
        return FALSE;
    while (*s) {
        if (isxdigit(*s) == FALSE)
            return FALSE;
        ++s;
    }
    return TRUE;
}
/*
 *
 * return true if s is a valid float
 *
 *
 */


int             isfloatnum(str)
    char           *str;
{
    char           *ptr;
    (void) strtod(str, &ptr);

    if (*ptr)
        return FALSE;
    else
        return TRUE;
}



/*
 * *    Function name : isaddress *
 *
 *      Description :  Determine wheter string is a valid address
 *
 * valid addressess:  1234,  #123AB *   Input :  A string *     Ouput : TRUE
 * or FALSE
 */
int             isaddress(s)
    char           *s;
{

    if (isdecnum(s) || (*s == '#' && !isspace(s[1]) && ishexnum(s + 1)))
        return TRUE;

    return FALSE;

}




int     parse_long(char *s, long *x, int assumed_base)
/* does not return a long */
{
/*  Recognized syntaxes:
    <hex_number>
    #<hex_number>
    <decimal_number>!
*/
   if (*s == '#') {
    if(ishexnum(s+1))
        sscanf(s, "#%lx", x);
    else
        return FALSE;
    }
   else
   if (s[strlen(s)-1]=='!') {
    if(isdecnum(strslc(s)))
        sscanf(s, "%ld!", x);
    else
        return FALSE;
    }
   else
    switch(assumed_base) {
    case 16:
            sscanf(s, "%lx", x);
            break;
    case 10:
            sscanf(s, "%ld", x);
            break;
    default:
            error("Invalid base (possible internal error).");
    }
   return TRUE;
}




int             isblank_line(s)
    char           *s;
{

    for (; *s; ++s)
        if (*s != ' ' && *s != '\t' && *s != '\n')
            return FALSE;

    return TRUE;

}



char           *parse_float(s, size_mantissa, size_exponent)
    char           *s;
    int             size_mantissa, size_exponent;
/* Returns NULL on invalid syntax */
/* Designed for the HP48 -> All half-bytes are reversed in the output */

/* (C) Alex Ramos 1993 */

#define INVALID_EXP (-7)        /* Any invalid  implicit exponent */

{
    char            mant[100];      /* char mant[size_mantissa + 1]; is
                                     * illegal */
    long            exp_exp = 0;    /* explicit exponent */
    int             imp_exp = INVALID_EXP,  /* implicit exponent (due to
                                             * decimal period) */
            sign = 1,       /* overall sign */
            exp_sign = 1,   /* exponent sign */
            lz_exp = 0,     /* Exponent implicited by leading
                     * zeroes */
            ptr;    /* linear parsing pointer */
    int             exparse = FALSE, nonzero = FALSE;
    *mant = '\0';

    ptr = -1;
    while (s[++ptr]) {
        switch (s[ptr]) {
        case 'E':
        case 'e':
            if (exparse)
                return NULL;
            exparse = TRUE;
            if (imp_exp == INVALID_EXP)
                imp_exp = ptr - 1;
            break;
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            nonzero = TRUE; /* notice no break here */
        case '0':
            if (exparse)
                exp_exp = exp_exp * 10 + atoi(chrtostr(s[ptr]));

            else if (s[ptr] != '0' || nonzero) {
                if (strlen(mant) < size_mantissa)
                    strcat(mant, chrtostr(s[ptr]));
                else
                    return NULL;
            } else
                --lz_exp;

            break;

        case '-':
            if (exparse)
                exp_sign = -1;
            else {
                sign *= -1;
                ++s;
                --ptr;
            }

            break;

        case '.':
            imp_exp = ptr - 1;
            break;

        default:
            return NULL;
        }
    }

    if (imp_exp == INVALID_EXP)
        imp_exp = ptr - 1;


    {

        char            out[100], fmt[20];
        int             i;
        long int        e = imp_exp + exp_sign * exp_exp + lz_exp;
        sprintf(fmt, "%c0%dld", '%', size_exponent);    /* e.g. "%05ld" */
        sprintf(out, fmt, e >= 0 ? e : long_pow(10L, (long) size_exponent) + e);
        strcpy(out, strrev(out));

        for (i = size_mantissa - 1; i >= 0; --i)
            strcat(out, chrtostr(i >= strlen(mant) ? '0' : mant[i]));
        strcat(out, chrtostr(sign == 1 ? '0' : '9'));

        return dupmark(out);
    }

}



char           *parse_ereal(s)
    char           *s;
/*
 * Example input: 3.14E-10 Example output: 099990000000000004130 Last digit
 * is sign: 0 or 9
 */
{
    char           *c;
    c = parse_float(s, 15, 5);      /* HP48's extended real */
    if (c != NULL)
        return c;
    error("Extended-real number expected.\n");
    return NULL;
}


char           *parse_real(s)
    char           *s;

{
    char           *c;
    c = parse_float(s, 12, 3);      /* HP48's single-precision real */
    if (c != NULL)
        return c;
    error("Real number expected.\n");
    return NULL;
}





typedef struct {
    char            c;
    char           *s;
}               CHAR_TAB;

static CHAR_TAB tab[] = {       /* "actual character", "backslash sequence" */
    {'"', "\\\""},          /* doublequote */
    {'\n', "\\n"},          /* newline */
    {'\\', "\\\\"},
    { 141, "\\->"},
    { 'A', "\\41"},
    {0, "\\\n"},            /* continue-next-line */
    {0, NULL}               /* end marker */
};


char           *str_unparse(s)  /* Untranslate character control codes */
    char           *s;
{
    char           *t = Malloc(strlen(s) * 2 + 2);
    *t = 0;
    while (*s) {
        int             i = 0;
        while (tab[i].s != NULL && tab[i].c != *s)
            ++i;

        if (tab[i].c) { /* *s matches i-th entry */
            strcat(t, tab[i].s);
            ++s;
        } else {        /* no match - no translation */
            char            cat[2];
            cat[0] = *s++;
            cat[1] = 0;
            strcat(t, cat);
        }
    }
    return mark(t);
}

char           * str_parse(s)   /* slow but works */
    char           *s;
/* Compile backslash sequences as real characters */
{
    char           *t = Malloc(strlen(s) + 2);
    *t = 0;
    while (*s) {
        if (*s == '\\') {
            int             i = 0;
            ++s;
            while ( (tab[i].s != NULL) &&
                strcmp(tab[i].s+1, str_sub(s, 0, strlen(tab[i].s)-2 )))
                ++i;
            if (tab[i].s != NULL) {
                strcat(t, chrtostr(tab[i].c));
                s += strlen(tab[i].s)-1;
                }
            else
                strcat(t, chrtostr(*s++));
        } else {
            strcat(t, chrtostr(*s++));
        }
    }

    return mark(t);
}


void    parse_args(char *s, char delim, char ***wordv, int *wordc)
{  
   /* New strategy: Malloc one large block that contains all strings.
      This large block can later be Freed by Freeing argv[0].  The 
      array of pointers should be Freed through its argv reference:
      { Free(wordv[0]); Free(wordv); *in calling function * }
   */  
   int    argc, i;   
   char **argv, *p;                      
   
   for(p=s, argc=1; *p; ++p) {
	if(*p == delim) ++argc;
   }         

   argv = Malloc(argc*sizeof(char *));
   p = argv[0] = Strdup(s);                     
   for(i=0; i<argc; ++i) {
      argv[i] = p;                             
      assert(p);
      p = strchr(p, delim);
      if (p) {
	*p = 0;
	++p;
      }
   }       
   (*wordv)=argv;
   (*wordc)=argc;                                           
}
         
                           
      
#if 0
{      
   int maxargs = 1;     /* dynamically resized */

   DEBUG(("parse_args('%s') invoked.\n", s));
   (*wordv) = (char **)Malloc(maxargs*sizeof(char **));
   (*wordc) = 0; 
   while(s!=NULL) {
        (*wordv)[*wordc]=Malloc(strlen(s)+1);
                                 /* wasteful Malloc to be fixed later */
        if(strchr(s, delim)==NULL)
            delim = '\0';        /* avoid overrun to missing delimiteter */
        memccpy((*wordv)[*wordc], s, delim, strlen(s)+1);
        (*strchr(((*wordv)[*wordc]), delim)) = '\0';
        (*wordv)[*wordc] = Realloc((*wordv)[*wordc],
                   strlen((*wordv)[*wordc]+1)); /* fix wasteful Malloc */
        s = delim ? strchr(s, delim)+1 : NULL;
        if (++(*wordc) >= maxargs)      /* too many args, resize */
            (*wordv)=(char **)Realloc(*wordv, (maxargs*=2)*sizeof(char **));
        }
   (*wordv) = (char **)Realloc(*wordv, (*wordc)*sizeof(char **));
}
#endif                 




char    *str_save(s)
    char    *s;

/* This function is simply a "long-term Strdup()"
   It Frees all memory when called with NULL; at the
   end of the program usually */

{
static  char **save=NULL;
static  int  mem=0, ptr=0;

if(s==NULL) {
    while(ptr)
        Free(save[--ptr]);
    Free(save);
    }

if(ptr==mem) {
    mem=2*(++mem);
    if(save==NULL)
        save = (char **)Malloc(mem*sizeof(char **));
    else
        save = (char **)Realloc(save, mem*sizeof(char **));
    }

return (save[ptr++] = Strdup(s));

}



int     file_size(filename)
char    *filename;
{
    struct stat statbuf;
    int     ioerror;

    ioerror = stat(filename, &statbuf);
    if (ioerror)
          error(nprintf( "Can't stat file %s.\n", filename));

    return statbuf.st_size;
}



/*
 *
 * Generic error-checking open file subroutine
 *
 *
 */

FILE           *file_open(char *path, char *filename, char *mode)
{

    FILE           *f;
    char           *fullname;
    fullname = Malloc(strlen(path) + strlen(filename) + 10);

    /* Discard the 'path' argument if filename appears to contain
       a full pathname. */
    if (filename[0] != '.' && filename[0] != '/') {
	    strcpy(fullname, path);
	    if (fullname[strlen(fullname) - 1] != '/')
		strcat(fullname, "/");
    } else {
	    *fullname = '\0';
    }

    DEBUG(("file_open() path=%s base=%s mode=%s\n",
            path, filename, mode));

    strcat(fullname, filename);

    DEBUG(("  fullname passed to fopen is '%s'\n", fullname));

    f = fopen(fullname, mode);
    if (f == NULL) {
        fprintf(stderr, "Can't open %s in \"%s\" mode: ", fullname, mode);
        perror(NULL);
        exit(1);
    }
                     
    Free(fullname);
    return f;
}


int		f_read(void *ptr, int size, int nitems, FILE *stream)
{
   /* An error-checking replacement for fread() */   
   int ioresult;  
   ioresult = fread(ptr, size, nitems, stream);   
   if(ioresult != nitems) {
      fprintf(stderr,
	      "fread() failed unexpectedly (sz=%d,ni=%d)\n",
	      size, nitems);
      abort();
   }
   return ioresult;
}


int		f_write(void *ptr, int size, int nitems, FILE *stream)
{
   /* An error-checking replacement for fwrite() */   
   int ioresult;  
   ioresult = fwrite(ptr, size, nitems, stream);   
   if(ioresult != nitems)
      perror("file write error");
   return ioresult;
} 


#ifdef PROVIDE_SPAWNVP

int	spawnvp(int mode, char *file, char **args) {
	if(mode==P_WAIT) {
		switch(fork()) {
			case -1: perror("fork failed.");
				 exit(1);	
			case  0: execvp(file, args);
				 break; 
			default:
				 wait(NULL);
			}
		}
	 else
		error("LIBRARY ERROR Non-implemented spawnvp() mode");
	 }

#endif

