
#define uses_stdlib
#include "config.h"

#include "symtab.h"
#include "error.h"
#include "input.h"
#include "globalv.h"
#include "misc.h"
#include "strlib.h"

/* Copyright (C) Alex Ramos 1993 */

/* Symbol-Table Utilities */

#define TRUE 1
#define FALSE 0

/*
Reminder on equal signs
    SASM considers a label that is preceded by an equal sign to be an
external label. The equal sign itself is not part of the name.
Equal signs here are thoroughly ignored - by function calls to strip_equal() in
add_any() and in lookup().
*/


int     hash(name)      /* Hashing function */
    char    *name;
{
   unsigned int h=0;
   while(*name)
    h = (h*4 + *name++) % (unsigned)HNUM;
   return (int)h;
}


static  void    init_hashtab()
{
   int i;
   static int hashtab_ready = FALSE;

   if (hashtab_ready == TRUE)
    fatal_error("init_hashtab() called twice... must be a bug.");

   hashtab_ready = TRUE;

   for(i=0; i<HNUM; ++i)   {
      hashtab[i] = -1;
      lasthash[i] = -1;   /*  separate to keep some compilers
                from complaining **/
      }
}


typedef struct  {
    char *label;
    char *legal;
    int   resolved;
    char *file;       /* First appearence, */
    int   line;       /* for error messages. */
 } UNTABLE ;


static UNTABLE *untable = NULL;
static int unres        = 0;
static int unres_mem    = 0;

#ifdef TRACE_ALL
void dump_untable()
{
   int i;
   DEBUG(("\n\n\ndump_untable()\n"));
   for(i=0; i<unres; ++i) {
     DEBUG(("label %s,  legal %s,   resolved %d\n",
            untable[i].label, untable[i].legal,
        untable[i].resolved));
   }
}
#endif


#ifdef UNIX
#	define POOLSIZE 512000
#else
#	define POOLSIZE (63L*1024L)    /* Below 64k on DOS! */
#endif

static char pool[POOLSIZE];
static char *poolptr=pool;


static char *estrdup(char *s)
   /* This function is a special Strdup() function. Its only purpose
   is to allocate pointers to a special memory pool which can be easily
   off-loaded to disk and back as a large block. Currently the only
   strings allocated in this fashion are in the global symbol table,
   thus providing for a faster startup of the program. */
{  
   char * const save = poolptr;
   long const u = poolptr-pool+strlen(s);
                                                                        
   if(u>=POOLSIZE) {                                       
         error(nprintf("Out of symbol space (%d bytes used).\n", u));              
         }                               
   while((*poolptr++ = *s++));
   return save;
}


void  savesym(char *file)  
{
   /* Save memory image of symbol table (data dump) */
   char        *base;
   size_t 	poolsize;
   FILE        *image;
   unsigned     deadbeef = 0xDeadBeef;

   if (!tablesize) {
       fatal_error("No input files specified (entry-point list required!)");
   }
   image = file_open(RPLPATH, file, "wb");
       
   note(nprintf("writing to \"%s\" on directory \"%s\".\n",
		file, RPLPATH));

   f_write(hashtab, sizeof(hashtab),  1, image);
   f_write(lasthash, sizeof(lasthash), 1, image);
   poolsize = (unsigned)(poolptr - pool);
   f_write(&poolsize, sizeof(size_t), 1, image);
   f_write(pool, 1, poolsize, image);
   f_write(&tablesize, sizeof(tablesize), 1, image);
   f_write(table, sizeof(table[0]), tablesize, image);
   base = &(pool[0]);
   f_write(&base, sizeof(char *), 1, image);
   f_write(&deadbeef, sizeof(deadbeef), 1, image);
   fclose(image);
}


void loadsym(char *file)   
{
   /* Load memory image */
   int      i;
   size_t   poolsize;                
   char    *oldbase;              
   FILE    *image = file_open(RPLPATH, file, "rb");
   unsigned checkbeef;

   f_read(hashtab, sizeof(hashtab), 1, image);
   f_read(lasthash, sizeof(lasthash), 1, image);
   f_read(&poolsize, sizeof(poolsize), 1, image);
   f_read(pool, poolsize, 1, image);
   f_read(&tablesize, sizeof(tablesize), 1, image);
   f_read(table, sizeof(table[0]), tablesize, image);
   f_read(&oldbase, sizeof(char *), 1, image);
   f_read(&checkbeef, sizeof(checkbeef), 1, image);
   if (checkbeef != 0xDeadBeef) {
	fatal_error("Corrupted image file (your fault)\n");
   }
   fclose(image);

   poolptr = pool + poolsize;
                             
   /* Relocate pointers to new image address -- yes this is 
      non-ANSI and "potentially unportable." Who cares? */
   for(i=0; i<tablesize; ++i) {
      table[i].name  = pool+((table[i].name)-(oldbase));
      table[i].value = pool+((table[i].value)-(oldbase));             
      }          
}

/*
 * add an entry to the symbol table

*/

/*

Relevant entry types:   'X' for a symble declared with EXTERNAL
        'd' for DEFINEs

*/


/* tablesize is an index to the next available slot
   in the global symbol table */


void    symbdef(eqsymbol, value, type)  /* Hash Table version , 3/20/93 */
        char   *eqsymbol, *value, type;
{
   char * symbol = strip_equal(eqsymbol);
   int  p, h = hash(symbol);

   DBSAMPLE(100,("symbdef('%s','%s','%c')\n",symbol,value,type));

if( (mode & LOADSYMBOL) || (p=lookup(symbol)) == -1) {
        /* Note - order of evaluation is relevant */
        /* Skip the redeclaration check if reading from entries files */
   if(hashtab[h] == -1)         /* First symbol with this hash value */
      hashtab[h]=tablesize;

   table[tablesize].name = estrdup(symbol);
   table[tablesize].value = estrdup(value);
   table[tablesize].type = type;
   table[tablesize].next = -1;

   if(lasthash[h] != -1)
      table[lasthash[h]].next = tablesize;
   lasthash[h] = tablesize;

   ++tablesize;
   }
else { /* redeclaration of symbol */                                                                   
   if( same(table[p].value, value))
       return;            /* Redeclaration with same value - harmless */

   warning(nprintf("%s redeclared as %s (was %s)\n",
        symbol, killblank(value), killblank(table[p].value)));
                                                                    
   Free(table[p].value);                
   table[p].value = Strdup(value);
   table[p].type = type;
   }
}


char    *legalize(word)
    char *word;
/* Generate  a new entry in the 'unresolved references' table
    Input:  An undefined or forward-referenced label.
    Output: A pointer to a legal name for the label.
    Parallel effect: Future references to this label will be replaced
        by its alias of the form "__unresXXXX"
        A pointer to static data is returned.
        external() depends on the latter assumption*/

{
   char leg[256];

   if(unres == unres_mem)
    if(untable)
        untable=(UNTABLE *)
            Realloc(untable,(unres_mem*=2)*sizeof(UNTABLE));
    else
        untable=(UNTABLE *)
            Malloc( (unres_mem=1)*sizeof(UNTABLE));


   untable[unres].label = Strdup(word);
   sprintf(leg, "_unr%x", unres);
   untable[unres].legal = Strdup(leg);

   sym_ilegal(word, leg);          /* 'i' for ilegal */
        /*
         * The 'i' is promptly replaced by 'X' when returning to
         * external(), so this assumes that EXTERNAL is the first
         * reference to that label in the program , i.e., the label
         * must have been fresh added and still the last in the table
         */
   untable[unres].resolved = FALSE;
   getstatus(source, &(untable[unres].file), &(untable[unres].line));
   return untable[unres++].legal;
}







char    *addtag(symbol)
    char *symbol;
/*
add a local tag entry to symbol table
A local tag entry is one of the form "label:" on a machine language block.
All tags are renamed to a legal alias (e.g. =#1_BLAH!@#$ becomes __unres00)
*/
{
   int  p;
   char *t;

/* Handles the case of resolution of a "true" forward reference */
/* where the legal name has already been assigned */
   if( (p = find_unres(symbol)) >= 0 ) {
    untable[p].resolved = TRUE;
    return untable[p].legal;
    }

/* Handles a normal label definition - not a forward reference, but generate
a new label anyway in case of illegal characters */

/* Note on 5/02/93 - does this ever execute? */

   t = legalize(symbol);
   untable[unres-1].resolved = TRUE;
   return t;

 }


static  void    init_builtin()  /* Define some "builtin" macros */
{

sym_define("LOCALLABEL", "LABEL");
sym_define("VISIBLE",   "COMMAND");
sym_define("ENDFCN",    "ABND");
sym_define("(WHILE)",   "::");

sym_equ("::", "#02D9D");
sym_equ(";",  "#0312B");
sym_equ("{",  "#02A74");
sym_equ("}",  "#0312B");
sym_equ("[#]","#02911");


/*
Functions have 3 nibble property fields. Also the default field for
commands is 8, for functions it is 000. For macro words the default
field is 0 though, so I guess these would be the definitions to use:

  - M.H. -

*/

sym_define("COMMAND",    "__imklib\nASSEMBLE\n\tCON(1) 8\nRPL\nNAME ");
sym_define("FUNCTION",   "__imklib\nASSEMBLE\n\tCON(3) 0\nRPL\nNAME ");
sym_define("EXPARSER",  "__imklib\nASSEMBLE\n\tCON(1) 0\nRPL\nNAME ");

/* External parser ? */

/*
These not yet. (Will incorporate all into the 3 above)

sym_define("xCOMMAND",   "ASSEMBLE\n\tCON(1) 8\nRPL\nxNAME ");
sym_define("sCOMMAND",   "ASSEMBLE\n\tCON(1) 8\nRPL\nsNAME ");
sym_define("hCOMMAND",   "ASSEMBLE\n\tCON(1) 8\nRPL\nhNAME ");

sym_define("xFUNCTION",  "ASSEMBLE\n\tCON(3) 0\nRPL\nxNAME ");
sym_define("sFUNCTION",  "ASSEMBLE\n\tCON(3) 0\nRPL\nsNAME ");
sym_define("hFUNCTION",  "ASSEMBLE\n\tCON(3) 0\nRPL\nhNAME ");

sym_define("xMACROWORD", "ASSEMBLE\n\tCON(1) 0\nRPL\nxNAME ");
sym_define("sMACROWORD", "ASSEMBLE\n\tCON(1) 0\nRPL\nsNAME ");
sym_define("hMACROWORD", "ASSEMBLE\n\tCON(1) 0\nRPL\nhNAME ");
*/
DEBUG(("Builtin symbols initialized.\n"));
}


/*
 * READTABLE: readtable(argc, argv)
 *
 * Read-in the symbol table  (called from rpp.c)
 *
 * The table consists of an HP label, address, type (supported, etc), and a
 * usage flag.
 */



void    readtable(argc, argv)
        int     argc;
        char    *argv[];
{

    FILE           *afile;
    int             i;

    DEBUG(("readtable(%d, '%s', ...)\n", argc, argv[0]));

    mode = LOADSYMBOL;
        /* skip the redeclaration check overhead - slight speed-up */

    if ( argc <= 0 ) {
        char *env = getenv("RPL_ENTRIES");
        if(env) {
            DEBUG(("fetched rplentries from environment: '%s'\n",env));
            parse_args(env, ' ', &argv, &argc);
        }
        else {
	    warning("No symbol tables specified. Use -t or set rplentries.\n");
        }
    }
    init_hashtab();
    DEBUG(("%d files will be read.\n", argc));
    for (i=0; i<argc; ++i) {
        char    aline[128];	/* safe (use fgets) */
        char    *ext,
                *fn=dupmark(argv[i]);              

    /* process each command line argument */

        DEBUG(("file %d, %s\n", i, fn));
        
    /* Take different action for ".e" or ".a" files */
        ext=strrchr(fn,'.');
        if( (ext&&samei(ext, ".ent")) || (ext&&samei(ext, ".e"))) {
            static loaded = 0;
            if(loaded)                                               
               error("Only ONE precompiled entries file can be loaded.\n");
            loaded=1;
            loadsym(fn);
            return;
        } else if(!ext || !samei(ext, ".a")) {
		warning("Entries file should have \".a\" extension.");
	}
           
         /* The following test must be done *after* it is known
         that we are not loading an image file. */
        if (tablesize == 0) {
           init_builtin(); /* No longer zero */
        }
        afile = file_open(RPLPATH, fn, "r");

        while (fgets(aline, sizeof(aline), afile) && !feof(afile)) {
            int iores;
            char symbol[sizeof(aline)+1], 
	         address[sizeof(aline)+1];
	   
	    if(strlen(aline)>=sizeof(aline)-1) {
		fatal_error("input line too long");
		abort();
	    }

        /* process each line in symbol file */

            iores = sscanf(aline, "%s EQU %s", symbol, address);

            if (iores == 2) {
                sym_equ(symbol, address);
            }
        }
        fclose(afile);
    }
    DEBUG(("Returning from readtable()\n"));
}


/*
 * int lookup(word)   :  Table lookup function
 *
 * word is an HP symbol from the .a file
 *
 * Returns: integer position in table, or a -1 if not found

 */


int     lookup(symbol)
    char    *symbol;
{
   char * symb = strip_equal(symbol);
   int h = hash(symb);
   int x = hashtab[h];
                
   DEBUG(("Looking up '%s'\n", symbol));
   while( x != -1 ) {
      if( same(table[x].name, symb) ) {
          DEBUG(("lookup('%s') returns %d\n", symb, x));
          return x;
          }
      x = table[x].next;
      }                                 
   DEBUG(("Not found!\n"));
   return -1;  /* not found */
}


SYMBOL  *slookup(name)
    char    *name;
/*
  Attempt to resolve a name, but do not force resolution.
*/
{
   int x;

   x = lookup(strip_equal(name));
   return x>0 ? &(table[x]) : NULL;
}

char    *try_resolve(name)
    char    *name;
{
   SYMBOL *sym=slookup(name);
   return sym==NULL ? NULL : sym->value;
}

/*

Symbol-resolve function

Resolves a symbol or expression

Returns a valid star expression

NOTE: Returns pointer provided by Malloc


*/

char
*atomic_resolve(symbol)    /* The cause of the 2-day bug! */
    char *symbol;

               /* if the symbol or # cannot be resolved,
                generate a forward-reference */
{
int pos;

DBSAMPLE(50,("atomic_resolve('%s')\n", symbol));

/* The "dot" special case is a workaround that should disappear
   when an actual expression PARSER is written */
if(same(symbol,".")) return(dupmark(symbol));

if(isaddress(symbol))
    return(dupmark(symbol));        /* Is this really a number? */

if( (pos = find_unres(symbol)) >= 0)
    return untable[pos].legal;      /* Is this a spotted
                       forward-reference? */

pos = lookup(symbol);

if(pos>=0) {
    DBSAMPLE(50, ("atomic_res Returning %s\n", table[pos].value));
    return dupmark(table[pos].value);
    }
else {
    char * t;
    t=legalize(symbol);
    DBSAMPLE(50, ("atomic_res Returning %s\n", t));
    /* should include an error-check here to verify if the symbol is so bad
       that not even SASM would accept it */
    return dupmark(t);
    }

}

/*
    A pseudo-parser to replace symbols within expressions. No parsing is done.
    A check for closing parenthesis, etc, would be approprate here.
*/

char    *resolve(arg)
    char *arg;

#define OPERS "+-/*!&^%"
{
   int i, j;
   char *e, *operand, *value, *expr;

   DEBUG(("char *resolve('%s');\n", arg));

   expr = dupmark(arg);
   e =    dupmark(arg);   /* two independent pointers are needed */


   /* First check out all parenthesis */
   for(i=0; i<strlen(e); ++i)
    if(e[i]=='(' || e[i]==')')
        e[i]=' ';



/*

Need to replace ! (operator) with |   :  NOT IMPLEMENTED

Need to replace * (operand) with .   :  NOT IMPLEMENTED



*/

  /* take care of the * operator.  More cases need to be included  */

 for(i=1; i<strlen(e)-1; ++i)
    if(e[i]=='*')
        if(expr[i-1]=='(' && expr[i+1]==')')
            expr[i]=e[i]='.';

   /* Now remove all operators (They'll be adjacent to spaces) */

   for(i=1; i<strlen(e)-1; ++i)
    if( strchr(OPERS, e[i] ) != NULL &&
             (e[i-1]==' '  || e[i+1]==' ') ) e[i]=' ';

   /* Now anything enclosed within spaces is a label or a number! */


   DEBUG(("resolve(): e:%s, expr:%s\n", e, expr));

   i=0; j=0;

   while(TRUE) {

   DEBUG(("top of while: i=%d, j=%d, e:%s\n", i, j, e));

   while(e[i]==' ')
      ++i;

   if(!e[i])
      break;

   j=i;

   while(e[j]!=' ' && e[j] )
      ++j;

   --j;


   operand = str_sub(e, i, j);

   value = atomic_resolve(operand);

   str_del(e, i, j);
   str_del(expr, i, j);

   DEBUG(("after str_del, e: %s, expr: %s\n", e, expr));

   e    = str_ins(e, value, i);
   expr = str_ins(expr, value, i);

   DEBUG(("after str_ins, e: %s, expr: %s\n", e, expr));

   i+=strlen(value);

   }

return expr;

}





int find_unres(word)
    char *word;
/*
Return label's position in the unresolved/illegal
table or a -1 if not there.
*/

{
   int i, r = -1;

   /* The table is negligibly small so linear search is not
      a problem. In one test this function was accounted for
      only 5% of the calls to strcmp() */

   for(i=0; i<unres; ++i)
    if(same(strip_equal(word), strip_equal(untable[i].label))) {
        r = i;
        break;
        }

   DEBUG(("find_unres(%s) returning %d\n", word, r));
   return r;
}


char    *get_legal_name(illegal)
    char *illegal;

/* This function designed specifically to be called from new_xlib */

{
  int p;
  char erm[100];
  char *c;

  p = find_unres(illegal);


  if (p>=0) {
        untable[p].resolved = TRUE;
        return untable[p].legal;    /* possible bug (static returned) */
        }


  sprintf(erm, "Library object %s was not declared as EXTERNAL.\n", illegal);
  note(erm);

  c = legalize(illegal);
  untable[unres-1].resolved = TRUE;
  return c;
}


void    list_unres()
{
	int i;
	for(i=0; i<unres; ++i)
	    if( ! untable[i].resolved )
		warning(nprintf(
		       "[line %d] Undefined symbol %s (%s).\n",
			untable[i].line,
			untable[i].label,
			untable[i].legal));
}


#ifdef TRACE_ALL
void    hashstat()
{
   int i, j, count, maxcount=0;
   int countcount[GLOBAL_TAB_SIZE];
   
   DEBUG(("BEGIN HASHSTAT\n"));
   for(i=0; i<GLOBAL_TAB_SIZE; countcount[i++]=0);
   for(i=0; i<=HNUM; ++i) {
      j = hashtab[i];
      count=0;
      while(j != -1)  {
        j = table[j].next;
        ++count;
        }

      if(count>maxcount)
        maxcount = count;

      DEBUG(("hash=%d\tcount=%d\n", i, count));
      ++countcount[count];
      }

   for(i=0; i<=maxcount; ++i) {
        DEBUG(("count= %d, \toccurrences= %d\n", i, countcount[i]));
   }
}
#endif


#ifdef TRACE_ALL
void    dump_gtable()
{
    int i;    
    DEBUG(("BEGIN GLOBAL SYMBOL TABLE\n"));
    for(i=0; i<tablesize; ++i)
    DEBUG(("%d: %s ==> %s (%c), h=%d, ptr=%d\n",
            i, table[i].name, table[i].value,
            table[i].type, hash(table[i].name),
            table[i].next));
    DEBUG(("END GLOBAL SYMBOL TABLE\n\n"));
}
#endif



char    *rplpath(p)
         char *p;
{
static  char    *path=".";

if(p)
    path=p;

DEBUG(("RPLPATH is now '%s'\n", path));

return path;
}



