
#ifndef TECHMEM_DISABLE	/* The whole module is transparently disableable.
			   the matching #endif is WAY at the bottom. */

char TECHMEM_RCS_ID[] =
"$Id: techmem.c,v 1.55 1994/05/05 22:46:20 ramos Exp $";

#include <sys/types.h>
#include <stdio.h>
#include <assert.h>
#include <malloc.h>

#define SUIT_malloc  malloc
#define SUIT_free    free
#define SUIT_realloc realloc
#define Assert(c,m)  assert(c)
#define max(a,b)	    (((a)>(b))?(a):(b))
	void LaTechCrash(x1,x2,x3,x4,x5,x6) /* Quick GCC Hack */ \
	{fprintf(stderr,x1,x2,x3,x4,x5,x6);abort();}

#ifndef TECHMEM_ALIGN
#define TECHMEM_ALIGN 8
#endif

#ifdef   UNIX
#include <sys/mman.h>
#    ifdef __sun__	/* no prototypes */
         void  mprotect(void *, int, int);
         void *valloc(int);
         int   getpagesize(void);
#    endif
#define  TECHMEM_PROT	/* Protect memory pages where appropriate. */
#endif

#define TECHMEM_16BIT

/***** End of configurable stuff. **********/

#define MAGIC	 0xDeadBeef
#define BAD_BYTE 0xEE /*Ideally this should cause a parity error or something*/

typedef	unsigned long int 	unlong; 

#ifndef NDEBUG
    static   unlong const    EndBlockSignature = 0xDeadBeef;
#   define   ADDROF_ENDSIG  (&EndBlockSignature)
#   define   SIZEOF_ENDSIG  sizeof(EndBlockSignature)
#else
#   define   SIZEOF_ENDSIG  0
#endif

/* Memory management library. A replacement for malloc() and etc.
 */

/* Terminology:
	A "public" block is the pointer that is known and used by the
application program. techmem_malloc returns a public block, and
techmem_free expects a valid public block.
	A "raw" block is a pointer to the Header that always precedes
the public block in memory. The application program never sees the
Header.
	Since the "public" block is simply a pointer to the data
following the header (a pointer to a raw block is also a pointer to
the header), the conversion is done by simply adding or subtracting
sizeof(Header) to the block pointer.  */

/* 
 *
 *
 *
 *      PRIVATE DEFINITIONS
 *
 *
 */


#ifndef NDEBUG	
  typedef struct {
      /* Block header that precedes all LaTech-managed public memory blocks
	 when in debugging mode. */
      unlong	magic;
      int	in_use;  /* 1 if in use, 0 if free. */
      unsigned 	size;    /* Size of the public block, in bytes. */
      unsigned 	blockid; /* This is a pointer into block list, for quick
			   location of the owner when freeing a block. */
      unlong    	checksum;
  } Header;
# define sizeof_Header (TECHMEM_ALIGN*(sizeof(Header)/TECHMEM_ALIGN+1))
#else
# define sizeof_Header 0
#endif /* !NDEBUG */

#define HEADER_OF(block) ((Header *) ( ((char *)(block)) - sizeof_Header ))
                /* A macro to "extract" the header information from a 
                   public block. The header is allways stored preceding
                   the publicly known block, i.e. at the top of the "raw"
                   malloc or SUIT_malloc block.*/

#define CHECKSUM(h) /* unlong CHECKSUM(Header *); */ \
        ((unlong)((h)->magic)^(h)->in_use^(h)->size^(h)->blockid^(unlong)(h))


#ifndef NDEBUG	/* Matching endif is WAY down there .. */

static void	magic_test(Header *header, char *function);

/* The indirect parameter passing procedure (provided by the 2 variables
   and the public function below) is used mainly for clarity (avoids
   cluttering the non-debug code with multiple prototypes).    */

static char	*owner_file=NULL, *owner_function=NULL;
static int       owner_line;

void	techmem_setowner(char *file, int line, char *function)
{
	/* Set the file name and line number corresponding to the next
	   call to techmem_malloc, which is about to happen. */
	owner_file 	= file;
	owner_line 	= line;
	owner_function 	= function;
}

typedef struct {
        unlong   checksum;
	void   *baseaddr;
	char   *file;
	int     line;
	char   *function;
} BLOCK_INFO;

#define BI_CHECKSUM(bip) /* unlong BI_CHECKSUM (BLOCK_INFO *); */\
        ( (unlong)((bip)->baseaddr) ^ (unlong)((bip)->file) ^ \
          (unlong)((bip)->line)     ^ (unlong)((bip)->function) ^ \
	  (unlong)(bip-list))

static  BLOCK_INFO *list      = NULL;
static	unsigned    listsize  = 0;
static  unsigned    listmem   = 0;
static  unsigned    listwatch = ~(0U);

#ifdef TECHMEM_PROT
void	protect_list(void)
{
    mprotect(list, 
	     listmem * sizeof(BLOCK_INFO), 
	     PROT_READ);
}

void	unprotect_list(void)
{
    mprotect(list, 
	     listmem * sizeof(BLOCK_INFO), 
	     PROT_READ|PROT_WRITE);
} 	
#else
#	define protect_list()
#	define unprotect_list()
#endif /* TECHMEM_PROT */

void	techmem_install_watchpoint(unsigned blockID)
{
	listwatch = blockID;
}

static	int	blocklist_append(void *rawblock)
{
	/* Called from techmem_malloc(), adds a block to the
	   list of allocated blocks. At this point the original
	   caller is known through the indirect parameter passing
	   done by techmem_setowner(). */

	if(listsize==listwatch) {
		LaTechCrash("Hit watchpoint at blockID=%d\n", listsize);
	}

	if(listsize==listmem) {
	    void *newlist;
	    int	  newsize;
#	    ifdef TECHMEM_PROT
	    newsize = listsize+getpagesize();
	    newlist = valloc (newsize*sizeof(BLOCK_INFO));
#	    else
	    newsize = listsize+1024;
	    newlist = SUIT_malloc (newsize*sizeof(BLOCK_INFO));
#	    endif
	    Assert(newlist,("really out of memory"));

	    if (list) {
		unprotect_list();
		memcpy(newlist, list, listsize*sizeof(BLOCK_INFO));
#	        ifdef TECHMEM_PROT
		free(list);
#	        else
		SUIT_free(list);
#		endif
	    }
	    list     = newlist;
	    listmem  = newsize;
	}

	unprotect_list();

	list[listsize].baseaddr = rawblock;
	list[listsize].file	= owner_file;
	list[listsize].line	= owner_line;
	list[listsize].function = owner_function;
	list[listsize].checksum = BI_CHECKSUM(&(list[listsize]));

	protect_list();

	return listsize++;
}

static void blocklist_remove(void *rawblock)
{
	/* Remove a block from the allocation list. */
	int	which_block;
	which_block = ((Header *)rawblock)->blockid;
	if (list[which_block].checksum != BI_CHECKSUM(&(list[which_block]))) {
	    LaTechCrash("Memory corruption: blocklist[%d] fails checksum\n"
			 "%08lx != (expect)%08lx\n", which_block,
			  list[which_block].checksum,
			  BI_CHECKSUM(&(list[which_block])));
	}
	unprotect_list();
	list[which_block].baseaddr = NULL; 		/* free'd */
	list[which_block].checksum = BI_CHECKSUM(&(list[which_block]));
	protect_list();
}

static  unlong    balmem, maxmem;

static void techmem_Account(long mem)
{
  /* Memory usage accounting */

  Assert(mem>0 || abs(mem)<=balmem,(/* Are we getting into negative mem.? */	 "Absurd assertion failure. PROBABLE MEMORY CORRUPTION BUG!!!!"));

  balmem += mem;
  maxmem = max(maxmem, balmem);
}
  
void	techmem_integrity_check(void)
{
    int	i;

    for(i=0; i<listsize; ++i) {
	if(list[i].checksum != BI_CHECKSUM(&(list[i])))
	    LaTechCrash("Bad: blocklist[%d] has been corrupted"
			" (%08lx != %08lx)",i,
			list[i].checksum, BI_CHECKSUM(&(list[i])));
	if(list[i].baseaddr)
	   magic_test(list[i].baseaddr, __FUNCTION__);
    }
}

void	techmem_print_block(int block_id)
{
    if (block_id<0)
	fprintf(stderr,   
		"Id #   Raw addr   Bytes        Owner:Line   Function\n");
    else
	fprintf(stderr,
		"%4d   %08lx   %5u %12s:%4d   %s\n",
		block_id,
		(long)(list[block_id].baseaddr),
		((Header *)(list[block_id].baseaddr))->size,
		list[block_id].file, 
		list[block_id].line, 
		list[block_id].function);	
}

int	techmem_reportMemoryUsage(void)
{
	/* This function should only be called at the end of execution,
	   and that's taken care of in LaTechDone. Returns non-zero if
	   there is a memory leak. */
	int i;

	techmem_integrity_check();

	fprintf(stderr, 
		"\nPeak memory usage:    %7lu bytes, %d blocks.\n"
		  "Current memory usage: %7lu bytes.\n",
		maxmem, listsize, balmem);

	if(balmem != 0) {
	    fprintf(stderr, "\n*** MEMORY LEAK DETECTED ***\n\n");
	    techmem_print_block(-1);
	    for(i=0; i<listsize; ++i) {
		if(list[i].baseaddr)
		    techmem_print_block(i);
	    }
	}
	return balmem!=0;
}


unsigned long techmem_usage(void)
{
    return balmem;
}

static void	magic_test(Header *header, char *function)
{
	/* Check to see if the memory block passed is a valid
	   memory block, by chekcing against the magic number.
	   IMPORTANT: If you get a segmentation fault in this
	   function, the problem is probably YOURS and not ours!
	   It will happen if you pass in a bogus pointer. */

    if (header==NULL)
	LaTechCrash("%s: NULL pointer passed in.", function);

    if (header->checksum != CHECKSUM(header)) {
	int 	i;
	for (i=0; i<listsize; ++i) {
	    if (list[i].checksum != BI_CHECKSUM(&(list[i]))) 
		LaTechCrash("Block failed checksum. "
			    "Blocklist failed checksum too.");
	    if (list[i].baseaddr == header)
		LaTechCrash("Poor memory block had its header smashed "
			    "by a mad pointer.");
	}
	LaTechCrash("%s: pointer %p is not a techmem memory block.",
		    function, header);
    }

    if (memcmp( (char *)header + sizeof_Header + header->size,
	       ADDROF_ENDSIG, SIZEOF_ENDSIG))
	LaTechCrash("%s: Memory corruption (no end signature in block).\n",
		    function);  

    if (header->magic != MAGIC)
	LaTechCrash("%s: bad magic number (weird, this really can't happen)", 
		    function);

    if (header->in_use == 0)
        LaTechCrash("%s: block was already free.\n", function);
}

#endif	/* !NDEBUG */



/*
 *
 *
 *
 *      PUBLIC FUNCTION DEFINITIONS
 *
 *
 *
 *
 *
 *
 *
 */

void *techmem_malloc(unsigned size)
{
  /* allocate a new LaTech Memory block a-la malloc */
	void *rawblock;

	rawblock = SUIT_malloc(size + sizeof_Header + SIZEOF_ENDSIG);
	
	if (rawblock == NULL)
		LaTechCrash("Out of memory (%u bytes requested)",size);

#       ifndef NDEBUG

#	ifdef TECHMEM_16BIT
	if (size>65535) {
	    LaTechCrash("block size above 65535 bytes is unportable");
	}
	if (size>32767) {
	    fputs("block size above 32767 bytes might be unportable",stderr);
	}
#	endif

        ((Header *)rawblock) -> size    = size;
        ((Header *)rawblock) -> magic   = MAGIC;
	((Header *)rawblock) -> in_use  = 1;
	((Header *)rawblock) -> blockid = blocklist_append(rawblock);
	((Header *)rawblock) -> checksum= CHECKSUM((Header *)rawblock);
	memcpy((char*)rawblock+sizeof_Header+size,ADDROF_ENDSIG,SIZEOF_ENDSIG);
	magic_test(rawblock, "**Impossible compiler or hardware bug!**");

	techmem_Account(size);
	memset( (char *)rawblock + sizeof_Header, '!', size);
	((char*)rawblock+sizeof_Header)[size-1]='\0';

#       endif

	return ((char *)rawblock) + sizeof_Header;
}


void  techmem_free(void *pblock)
{
  /* Free a previously allocated memory block. 
     Has no effect if parameter is NULL.
     Crashes if the block fails magic number test.
     Updates the memory accounting function.
     Fills block with an easily recognizeable pattern.*/

  if(pblock!=NULL) {
#	ifdef NDEBUG
        SUIT_free(pblock);
#	else
	Header  *hdr = HEADER_OF(pblock);
	magic_test(hdr, "LaTechFree");
  	blocklist_remove(hdr);
  	techmem_Account( hdr->size * (-1L) );

	hdr->in_use = 0;
	hdr->checksum = CHECKSUM(hdr);
	memset(pblock, BAD_BYTE, hdr->size);

	SUIT_free(hdr);
#       endif
  }
}



/*
 *
 * Nicer interface functions (all use the above two).
 *
 * these work pretty much a-la Standard malloc lib
 *
 */


void *techmem_realloc(void *oldblock, unsigned newsize)
{
/* The debugging version of this function is very innefficient.
   The non-debugging version is quite adequate however.*/

	void *newblock;

# ifndef NDEBUG
	size_t oldsize;

	if(oldblock==NULL) {
		return techmem_malloc(newsize);
	}
	else {
	        magic_test(HEADER_OF(oldblock), "LaTechRealloc");

		oldsize = HEADER_OF(oldblock) -> size;

		if(newsize <= oldsize) {
			/* Realloc of smaller block has no effect */
			return oldblock;
		}

		newblock = techmem_malloc(newsize);

		if(oldsize)
			memcpy(newblock, oldblock, oldsize);

		techmem_free(oldblock);
		return newblock;
	}
# else
	if (oldblock)
	    newblock = SUIT_realloc(oldblock, newsize);
	else
	    newblock = SUIT_malloc(newsize);
	
	if(newblock==NULL)
	    LaTechCrash("Out of memory in realloc (%d bytes)", newsize);

	return newblock;
# endif
}



char *techmem_strdup (char *s)
{
	char *copy;
	Assert(s != NULL,("LaTechStrdup was passed a NULL pointer"));
	copy = techmem_malloc(strlen(s) + 1);
	strcpy(copy, s);
	return copy;
}



void *techmem_calloc (unsigned nelem, unsigned elsize)
{
	void *array;
	array = techmem_malloc(nelem*elsize);
	memset(array, 0, nelem*elsize);
	return array;
}

#endif  /* ! TECHMEM_DISABLE */

/* $Log: techmem.c,v $
 * Revision 1.55  1994/05/05  22:46:20  ramos
 * Added some cross-project portability (-DTECHMEM_GENERIC)
 *
 * Revision 1.54  1994/04/19  03:44:25  ramos
 * Minor change: watchpoints call LaTechCrash (not Warning).
 *
 * Revision 1.53  1994/04/02  02:14:44  ramos
 * Added "number of blocks" statistic to report/
 *
 * Revision 1.52  1994/04/02  02:09:48  ramos
 * Bugfix (bad one): When something moves around in memory (as in realloc),
 * using it's address as part of the checksum is not a good idea.
 *
 * Revision 1.51  1994/03/31  06:38:12  ramos
 * Memory protection is used in the blocklist when compiling under UNIX.
 *
 * Revision 1.50  1994/03/31  05:22:10  ramos
 * Unstable check-in before more changes.
 *
 * Revision 1.49  1994/03/31  03:42:42  ramos
 * LaTechWarning replaced by fputs().
 *
 * Revision 1.48  1994/03/31  03:29:24  ramos
 * Improved magic_test() differentiates between "corrupted" and "bogus"
 * block pointers.
 * Added 16-bit portability checks.
 *
 * Revision 1.47  1994/03/31  02:15:45  ramos
 * Made techmem_integrity_check() into its own.
 *
 * Revision 1.46  1994/03/27  11:17:13  ramos
 * Modified some misleading error messages.
 *
 * Revision 1.45  1994/03/25  07:28:49  ramos
 * Lotsa whitespace modifications trying to fix a bug which was later
 * located in techmem.h.
 *
 * Revision 1.44  1994/03/25  03:05:40  ramos
 * Implemented memory corruption checks. Seem to work.
 *
 * Revision 1.43  1994/03/24  20:40:47  ramos
 * Temporary checking, need the old version to compile something...
 *
 * Revision 1.42  1994/03/17  03:30:50  ramos
 * No need to include <signal.h>.
 *
 * Revision 1.41  1994/03/10  15:42:37  ramos
 * Minor fix for MS-Windows port.
 *
 * Revision 1.40  1994/02/28  21:00:49  ramos
 * Removed the damn signal handler altogether. !
 *
 * Revision 1.39  1994/02/28  05:54:23  ramos
 * No, NOW fixed bug in signal handler.
 *
 * Revision 1.38  1994/02/28  05:26:04  ramos
 * Fixed bug in signal handler.
 *
 * Revision 1.37  1994/02/24  05:54:57  ramos
 * Removed an unnecessary setjmp/longjmp obfuscation.
 *
 * Revision 1.36  1994/02/24  05:27:54  ramos
 * Syntax corrections.
 *
 * Revision 1.35  1994/02/24  05:26:04  ramos
 * Simple speed/efficiency hack: set the signal handler only once,
 * then test internally.
 *
 * Revision 1.34  1994/02/24  05:03:04  ramos
 * Borland C++ compatibility change -- avoid zero-sized struct on NDEBUG.
 *
 * Revision 1.33  1994/02/19  07:38:51  dgarrett
 * Adjusted to new asserts.
 *
 * Revision 1.32  1994/01/10  00:04:48  ramos
 * Implemented techmem_usage().
 *
 * Revision 1.31  1994/01/05  19:15:29  ramos
 * Better error message for Out Of Memory.
 *
 * Revision 1.30  1994/01/05  15:50:08  ramos
 * latech.h searched and replaced by techpriv.h in all modules
 *
 * Revision 1.29  1994/01/04  22:34:01  ramos
 * added magic test to LaTechRealloc
 *
 * Revision 1.28  1993/11/04  19:20:54  ramos
 * Maintain 'out of memory' test in the NDEBUG version of LaTechRealloc.
 *
 * Revision 1.27  1993/11/04  19:07:28  ramos
 * Don't rely on undocumented SUIT_realloc's accepting a NULL argument.
 *
 * Revision 1.26  1993/10/16  18:25:08  ramos
 * Corrected a small typo.
 *
 * Revision 1.25  1993/10/16  04:22:04  ramos
 * Corrected innacurate #ifdef leftover from obsolete debug levels.
 *
 * Revision 1.24  1993/10/16  02:57:32  ramos
 * General code cleanup (removed obsolete/neverused functions).
 * Modified techmem_realloc to make use of the undocumented SUIT_realloc().
 * Reduced multiple levels of debugging to 3 (normal, NDEBUG, and DISABLE).
 *
 * Revision 1.23  1993/10/05  23:05:31  ramos
 * The double-freeing detection was made operational (didn't work
 * before due to a small bug).
 *
 * Revision 1.22  1993/10/05  22:12:59  ramos
 * Added assertion for positive memory ballance only.
 *
 * Revision 1.21  1993/09/24  00:31:52  ramos
 * Added watchpoint feature for aborting whenever a pre-specified
 * block ID number is reached. (doing this with a debugger would have
 * been unusably slow).
 *
 * Revision 1.20  1993/09/23  05:13:10  dgarrett
 * Got rid of LaTechSprintf.
 *
 * Revision 1.19  1993/09/11  23:35:45  ramos
 * output format changed
 *
 * Revision 1.18  1993/09/11  22:13:53  ramos
 * stores function name with block ownership.
 *
 * Revision 1.17  1993/09/11  21:50:15  ramos
 * memusage report returns an error code (non-zero if leak is detected).
 *
 * Revision 1.16  1993/09/11  21:41:31  ramos
 * \t
 *
 * Revision 1.15  1993/09/11  21:39:53  ramos
 * Added "Serial number" to the memory blocks in techmem_report (no
 * modifications to other code).
 *
 * Revision 1.14  1993/09/08  06:11:33  dgarrett
 * Modified CONST back to const.
 *
 * Revision 1.13  1993/09/08  04:33:31  dgarrett
 * Moved log entries to end of file.
 *
 * Revision 1.12  1993/09/08  01:55:22  dgarrett
 * Updated RCS_ID to be NAME_RCS_ID and none-static. Gets rid of warning.
 *
 * Revision 1.11  1993/09/06  16:40:39  ramos
 * Added a signal handler for nice error messages (in case the
 * block passed to Free is not trusteable).
 *
 * Revision 1.10  1993/09/06  12:10:52  dgarrett
 * Modified const to const so it can be defined to const or '' depending on the
 * compiler.
 * All files using stdargs were adjsuted slightly.
 *
 * Revision 1.9  1993/09/03  04:21:07  dgarrett
 * Got rid of unused buffer.
 *
 * Revision 1.8  1993/09/03  02:33:57  dgarrett
 * Got rid of Assert warnings about pointers.
 *
 * Revision 1.7  1993/08/31  00:25:44  ramos
 * Added 16-byte alignment restriction.
 * Fixed a bug on the blocklist allocation.
 *
 * Revision 1.6  1993/08/30  23:12:20  ramos
 * Added "disable" option.
 *
 * Revision 1.5  1993/08/30  18:33:54  ramos
 * Added memory leak detection (block ownership).
 * Implemented Sprintf, but it's a bad algorithm.
 *
 * Revision 1.4  1993/08/27  00:48:41  ramos
 * New latech_reportMemoryUsage for detecting memory leaks.
 *
 * Revision 1.3  1993/08/26  07:19:24  ramos
 * Added comments.
 * Removed some cluttering debug messages.
 *
 * Revision 1.2  1993/08/26  07:03:07  ramos
 * Added extra parenthesis to a dubious precedence resolve.
 * Added "magic number" for detecting bad memory blocks.
 *
 * Revision 1.1  1993/08/26  06:36:56  ramos
 * Initial revision
 *
 */



