/*
	Pong.c
	Version:	1.0
	Written By: Joshua Bogue
	Email:		jammer2000b@gmail.com
	Date:		  8/17/07
	License: GPLv2
	
	This was written with HPGCC to practice programming in C and to add another
	game for the HP 50G, though should work for 49G+  This is the classic
	Pong game with one difference: The only real way to beat the computer is
	to have the paddle moving with in the direction of the ball as the paddle
	hits the ball, which will speed up the ball.  Once you do this twice the
	ball should be moving fast enough to get out of the computers range a lot
	of the time, but for a challenge speed the ball up more.  Let me know if
	you find any bugs or have any fixes (or even a cool graphical screen to
	add to the game :-) ).  This game has been tested with the HP50G only,
	and I don't have another calculator to test with.  Though I have tried to
	find the bugs, this software may freeze your calculator, so be
	careful with it.  Have fun and program with HPGCC if you get a chance :-)
*/
	
#include <hpgcc49.h>
#include <hpgraphics.h>

void initializeDisp(void);
void DrawArena(void);
int DrawUser(int location);
int DrawComputer(int location);
void DrawBall(void);
int calcCompPosition(int location);
void adjustSpeed(void);
void printSignedNum(int numToPrint, int x, int y, int numMaxLength);

#define ARENA_WIDTH 130
#define ARENA_HEIGHT 71
#define ARENA_TOP 8
#define ARENA_CENTER_X 65
#define ARENA_CENTER_Y 41
#define LEFT 0
#define RIGHT 1
#define UP -1 // Defined as when the ball is going towards top of calc. screen
#define DOWN 1 // Vice Versa
// There is a pixel offset to each shape you are using
// Only really needed during collision detection...
#define SHAPE_OFFSET 1

// Lazy method variable declarations
static int userScore = 0;
static int compScore = 0;
const static int MAX_SCORE = 10;
static int userSide = LEFT;
static int compSide = RIGHT;
static int paddleHeight = 6;
static int paddleWidth = 2;
static int paddleOffset = 1;
static int ballXcenter = ARENA_CENTER_X;
static int ballYcenter = ARENA_CENTER_Y;
static int ballXstep = 1;
static int ballYstep = 1;
static int ballRadius = 3;
static int computerDelay = 2;
static int delayCount = 0;
static int currCompPaddle = ARENA_CENTER_Y;
static int currUserPaddle = ARENA_CENTER_Y;
static int speedBoost = 0;

int main(void)
{
    initializeDisp();
    
	while (!keyb_isON() && userScore < MAX_SCORE && compScore < MAX_SCORE)	// wait until ON pressed
    {
		DrawArena();		// Draw the actual arena
		currUserPaddle = DrawUser(currUserPaddle);
		currCompPaddle = DrawComputer(currCompPaddle);
		DrawBall();
		hpg_flip();		// Show the drawn on screen...
		hpg_clear();
		sys_LCDSynch();		// wait for LCD Refresh
		if (keyb_isUp()) {
			currUserPaddle -= paddleHeight;
			speedBoost = UP;
		}
		else if (keyb_isDown()) {
			currUserPaddle += paddleHeight;
			speedBoost = DOWN;
		}
		else
		{
			speedBoost = 0;
		}
		
		//sys_waitRTCTicks(1);  // Uncomment this to slow game down for debugging
		
		// Choose a new position for ball
		ballXcenter += ballXstep;
		ballYcenter += ballYstep;
		currCompPaddle = calcCompPosition(currCompPaddle);
		//int i = 0;
		//for (; i < 1000000; i++) {}
    }
	// If the user has won...
	if(userScore == MAX_SCORE)
	{
		hpg_draw_text("USER WINS!!!", 45, 38);
		hpg_draw_text("Press ON to exit", 35, 44);
		hpg_flip();
		while (!keyb_isON()) {}
	}
	// If the comuter has won...
	else if(compScore == MAX_SCORE)
	{
		hpg_draw_text("COMPUTER WINS!!!", 35, 38);
		hpg_draw_text("Press ON to exit", 35, 44);
		hpg_flip();
		while (!keyb_isON()) {}
	}

    return 0;
}

/*
	Method to initialize the LCD to desired settings.
*/
void initializeDisp(void)
{
    hpg_set_mode_gray4(1);
    hpg_clear();
}

/*
	Method to draw the basic arena with no paddles or ball.
*/
void DrawArena(void)
{
    // Draw the top header
    hpg_fill_rect(0, 0, ARENA_WIDTH, ARENA_TOP - 1);
    hpg_set_color(hpg_stdscreen, HPG_COLOR_WHITE);
    hpg_set_font(hpg_stdscreen, hpg_get_bigfont());
    hpg_draw_text("PONG", 1, 1);
    hpg_set_font(hpg_stdscreen, hpg_get_minifont());
    hpg_draw_text("User:", 36, 2);
    hpg_draw_text("Comp:", 66, 2);
	hpg_draw_text("Spd:", 96, 2);
	printSignedNum(userScore, 56, 2, 3);
	printSignedNum(compScore, 86, 2, 3);
	printSignedNum(abs(ballXstep), 116, 2, 3);
    // Draw the actual arena that the ball travels in
    hpg_set_color(hpg_stdscreen, HPG_COLOR_BLACK);
    hpg_draw_rect(0, ARENA_TOP, ARENA_WIDTH, 79);
}

/*
	Checks to make sure that the user's paddle is inside of reasonable
	boundaries.  If it is then the user's paddle is drawn, otherwise it is
	placed into the closest reasonable spot.
*/
int DrawUser(int location)
{
    int minY = ARENA_TOP + 1;
    int maxY = 78;

    if (location - paddleHeight <= minY)	// You hit the top of the arena
    {
		location = minY + paddleHeight + 1;
    }
	else if (location + paddleHeight >= maxY)	// You hit the bottom of the arena
    {
		location = maxY - paddleHeight - 1;
    }
    if (userSide == 0) {
		hpg_draw_rect(2, location - paddleHeight, 2 + paddleWidth, location + paddleHeight);
    }
	else
	{
		hpg_draw_rect(ARENA_WIDTH - paddleWidth - 2, location - paddleHeight, ARENA_WIDTH - 2, location + paddleHeight);
    }

    return location;		// Return because this method can change the location
}

/*
	Checks to make sure that the computer's paddle is inside of reasonable
	boundaries.  If it is then the user's paddle is drawn, otherwise it is
	placed into the closest reasonable spot.
*/
int DrawComputer(int location)
{
    int minY = ARENA_TOP + 1;
    int maxY = 78;

    if (location - paddleHeight <= minY)	// Computer hit the top of the arena
    {
		location = minY + paddleHeight + 1;
    }
	else if (location + paddleHeight >= maxY)	// Computer hit the bottom of the arena
    {
		location = maxY - paddleHeight - 1;
    }
    if (compSide == 0) {
		hpg_draw_rect(2, location - paddleHeight, 2 + paddleWidth, location + paddleHeight);
    }
	else
	{
		hpg_draw_rect(ARENA_WIDTH - paddleWidth - 2, location - paddleHeight, ARENA_WIDTH - 2, location + paddleHeight);
    }

    return location;
}

/*
	Allows the computer's paddle to be put in its next spot based on where
	the ball is.
*/
int calcCompPosition(int location)
{
    if (delayCount == computerDelay)
	{
		if (ballYcenter < location)
		{
			location -= (paddleHeight / 2) + 1;
		}
		else if (ballYcenter > location)
		{
			location += (paddleHeight / 2) + 1;
		}
		delayCount = 0;
    }
	else
	{
		delayCount++;
    }

	return location;
}

/*
	Currently this function first checks if the ball has collided with either
	computer/player paddle or has collided with the arena.  If the ball has
	collided with anything, then the position of the ball is fixed, so that
	it doesn't look like the ball is in the wall, or paddle.  If the ball has
	gotten past a paddle and collided with a wall the opposite player gets a
	point.  Finally the ball is drawn in a new spot.
*/
void DrawBall(void)
{
    int minY = ARENA_TOP;
    int maxY = 79;
    int minX = 0;
    int maxX = ARENA_WIDTH + 1;
	
    // Check to see if the ball has hit a paddle
    // Is who is on the LEFT side? Let's deal with them first...
    // If it is the Computer...
    if ((ballYcenter >= currCompPaddle - paddleHeight)
	&& (ballYcenter <= currCompPaddle + paddleHeight)
	&& (ballXcenter - (minX + ballRadius + paddleWidth + SHAPE_OFFSET * 2 + paddleOffset)) <= 0
	&& (compSide == LEFT))
	{
		ballXcenter = minX + ballRadius + paddleWidth + paddleOffset + SHAPE_OFFSET * 2;
		ballXstep = -ballXstep;
		adjustSpeed();
    }
    // If it is the Player...
    else if ((ballYcenter >= currUserPaddle - paddleHeight)
	     && (ballYcenter <= currUserPaddle + paddleHeight)
	     && (ballXcenter - (minX + ballRadius + paddleWidth + SHAPE_OFFSET * 2 + paddleOffset)) <= 0
	     && userSide == LEFT)
	{
		ballXcenter = minX + ballRadius + paddleWidth + paddleOffset + SHAPE_OFFSET * 2;
		ballXstep = -ballXstep;
		adjustSpeed();
    }
    // Is who is on the RIGHT side? Let's deal with them first...
    // If it is the Computer...
    else if ((ballYcenter >= currCompPaddle - paddleHeight)
	     && (ballYcenter <= currCompPaddle + paddleHeight)
	     && ((maxX - ballRadius - paddleWidth - SHAPE_OFFSET * 2 - paddleOffset) - ballXcenter) <= 0
	     && compSide == RIGHT)
	{
		ballXcenter = maxX - ballRadius - paddleWidth - paddleOffset - SHAPE_OFFSET * 2;
		ballXstep = -ballXstep;
		adjustSpeed();
    }
    // If it is the User...
    else if ((ballYcenter >= currUserPaddle - paddleHeight)
	     && (ballYcenter <= currUserPaddle + paddleHeight)
	     && ((maxX - ballRadius - paddleWidth - SHAPE_OFFSET * 2 - paddleOffset) - ballXcenter) <= 0
	     && userSide == RIGHT)
	{
		ballXcenter = maxX - ballRadius - paddleWidth - paddleOffset - SHAPE_OFFSET * 2;
		ballXstep = -ballXstep;
		adjustSpeed();
    }
    // Check to see if the ball has reached the left or right of the arena
    else if (ballXcenter - ballRadius <= minX)
	{
		ballXcenter = minX + ballRadius + 1;
		ballXstep = -ballXstep;
		if (userSide == 1) {
			userScore++;
		}
		else
		{
			compScore++;
		}
    }
	else if (ballXcenter + ballRadius >= maxX)
	{
		ballXcenter = maxX - ballRadius - 1;
		ballXstep = -ballXstep;
		if (userSide == 0) {
			userScore++;
		}
		else
		{
			compScore++;
		}
    }
    // Check to see if the ball has reached the top or bottom of the arena
    if (ballYcenter - ballRadius <= minY)	// ball hit the top of the arena
    {
		ballYcenter = minY + ballRadius + 1;
		ballYstep = -ballYstep;
	}
	else if (ballYcenter + ballRadius >= maxY)	// Ball hit the bottom of the arena
    {
		ballYcenter = maxY - ballRadius - 1;
		ballYstep = -ballYstep;
    }

    hpg_fill_circle_on(hpg_stdscreen, ballXcenter, ballYcenter,  ballRadius);
}

/*
	Adjust the balls speed based on if the paddle is currently moving in a
	direction that is with or against the ball.  Faster for same direction,
	Slower for opposite direction.
*/
void adjustSpeed(void)
{
	if(ballXstep < 0 && speedBoost == UP)
	{
		ballXstep += speedBoost;
	}
	else if(ballXstep > 0 && speedBoost == UP)
	{
		ballXstep -= speedBoost;
	}
	if(ballYstep > 0 && speedBoost == DOWN)
	{
		ballYstep += speedBoost;
	}
	else if(ballYstep < 0 && speedBoost == DOWN)
	{
		ballYstep -= speedBoost;
	}
}

/*
	Handy function to take a number and print it as a string to the screen
	at the given x and y position.
*/
void printSignedNum(int numToPrint, int x, int y, int numMaxLength)
{
	if(numMaxLength < 2) { numMaxLength = 20; }
	if(x < 0) { x = abs(x); }
	if(y < 0) { y = abs(y); }
	
	char* buffer1 = malloc(numMaxLength + 1);
	hpg_draw_text(itoa(numToPrint, buffer1, 10), x, y);
	free(buffer1);
}

