/*
Steering Behaviors For Autonomous Characters by Craig W. Reynolds: http://www.red3d.com/cwr/steer/gdc99/

the program implements three of the steering behaviors described in this paper:
	seek with arrival
	wander with collision avoidance
	path following
	
right arrow key to cycle through the behaviors and on key to quit	
	
version 1.01 - corrects a bug in wander() that made a agent disappear when the absolute value of angle was below 0.01 but different from 0	
	
*/

#include <hpstdlib.h>
#include <hpkeyb49.h>
#include <hpmath.h>
#include <hpgraphics.h>
#include <hpstdio.h> 

#define DEGTORAD 0.017453293

#define WIDTH 131
#define HEIGHT 80

#define AGENTS  10
#define SEEKAGENTS 2
#define MAXPATHPOINTS 3

typedef struct
{ float x; float y; } _2dvector;

_2dvector AgentShape[3] = { {5.0,0.0}, {-5.0,3.0}, {-5.0,-3.0} };

typedef struct
{
	_2dvector position;
	_2dvector heading;
	_2dvector shape[3];
	float rotation;
	_2dvector scale;
	float radius;
} _agent;

typedef struct
{
	_2dvector target;
	int flag;
}_seekinfo;

typedef struct
{
	_2dvector target;
	float radius;
	float distance;
	float speed;
}_wanderinfo;

_agent Agents[AGENTS];
_wanderinfo WanderAgents[AGENTS];
_seekinfo SeekAgents[AGENTS];
_2dvector screencenter = { WIDTH/2.0 , HEIGHT/2.0 };

void InitialiseAgents(void);
void InitialiseSeek(void);
void InitialiseWander(void);
void draw_agent(_agent *agent);
int seek(void);
int wander(void);
void nooverlap(int);
int pathfollowing(void);
void steeringbehaviors(void);

float _2dvecDot(_2dvector *v1, _2dvector *v2);
void _2dvecNormalize(_2dvector *v);
int _2dvecSign(_2dvector *v1, _2dvector *V2);

int main(void)
{
	unsigned char ticks = sys_setRTCTickPeriod(1);
	hpg_set_mode_mono(1);
	srand(0);
	steeringbehaviors();
	sys_setRTCTickPeriod(ticks); 
	return 0;
}

void steeringbehaviors(void)
{
	for(;;) {
		InitialiseSeek();
		InitialiseAgents();
		if(seek()) return;
		InitialiseWander();
		InitialiseAgents();
		if(wander()) return;
		Agents[0].scale.x = 0.5; Agents[0].scale.y = 0.5;
		InitialiseAgents();
		if(pathfollowing()) return; 
	}
}

void InitialiseAgents(void)
{
	int i,j;
	float start_rotation;
	
	for(i = 0; i < AGENTS; i++) {		
		start_rotation = (float) (rand() % 360) * DEGTORAD;
		
		Agents[i].radius = AgentShape[0].x;
		
		Agents[i].position.x = (float) (rand() % 100) - 50; 
		Agents[i].position.y = (float) (rand() %  50) - 25; 
		Agents[i].rotation = 0.0;
		
		for (j=0 ; j<3 ;j++) {	
			Agents[i].shape[j].x = ((AgentShape[j].x * Agents[i].scale.x) * cos(start_rotation)) - 
									((AgentShape[j].y * Agents[i].scale.x) * sin(start_rotation));
			
			Agents[i].shape[j].y = ((AgentShape[j].x * Agents[i].scale.y) * sin(start_rotation)) + 
									((AgentShape[j].y * Agents[i].scale.y) * cos(start_rotation));		
		}
		Agents[i].heading.x = Agents[i].shape[0].x; 
		Agents[i].heading.y = -Agents[i].shape[0].y; 
		_2dvecNormalize(&Agents[i].heading);
	}
}

void draw_agent(_agent *agent)
{
	int agent2dx[3], agent2dy[3];
	int i;
	_2dvector as;
	
	for (i=0 ; i<3 ;i++) {	
		as.x = (agent->shape[i].x * cos(agent->rotation)) - (agent->shape[i].y * sin(agent->rotation));
		as.y = (agent->shape[i].x * sin(agent->rotation)) + (agent->shape[i].y * cos(agent->rotation));
		
		agent->shape[i].x = as.x; 
		agent->shape[i].y = as.y;
		
		agent2dx[i] = (int) screencenter.x + agent->position.x + as.x;
		agent2dy[i] = (int) screencenter.y - agent->position.y + as.y;
	}
	agent->heading.x = agent->shape[0].x;
	agent->heading.y = -agent->shape[0].y;
	_2dvecNormalize(&agent->heading);

	hpg_fill_polygon(agent2dx, agent2dy, 3);
}

int seek(void)
{
	float angle = 0.0;
	_2dvector totarget;
	int i;
	
	for(;;) {
		hpg_clear();
		for(i = 0; i < SEEKAGENTS; i++) {
			hpg_draw_circle(screencenter.x + SeekAgents[i].target.x,screencenter.y - SeekAgents[i].target.y, 4);
			draw_agent(&Agents[i]);
			
			totarget.x = (SeekAgents[i].target.x - Agents[i].position.x) * 5.0;
			totarget.y = (SeekAgents[i].target.y - Agents[i].position.y) * 5.0;
			
			if(_fabs(totarget.x) >= 3.0 ) Agents[i].position.x += totarget.x * 0.04;
			if(_fabs(totarget.y) >= 3.0 ) Agents[i].position.y += totarget.y * 0.04;
			if( SeekAgents[i].flag == 0 && _fabs(totarget.x) < 3.0 && _fabs(totarget.y) < 3.0 ) {
				SeekAgents[i].flag = 1;
				SeekAgents[i].target.x = (float) (rand() % 110) - 50;
				SeekAgents[i].target.y = (float) (rand() % 70) - 35;
				totarget.x = (SeekAgents[i].target.x - Agents[i].position.x) * 5.0;
				totarget.y = (SeekAgents[i].target.y - Agents[i].position.y) * 5.0;
			}
			if(SeekAgents[i].flag) {
				_2dvecNormalize(&totarget);
				angle = _2dvecSign(&Agents[i].heading, &totarget) * acos(_2dvecDot(&Agents[i].heading, &totarget) );
				_fabs(angle) > 0.1 ? (Agents[i].rotation = angle/2.0) : (Agents[i].rotation = 0.0, SeekAgents[i].flag = 0);
			}
		}
		hpg_flip(); sys_waitRTCTicks(8);
		if(keyb_isRight()) return 0;
		if(keyb_isON()) return 1;
	}
}

int wander(void)
{
	float angle = 0.0;
	_2dvector totarget;
	int i;
	
	for(;;) {
		hpg_clear();
		
		for(i = 0; i < AGENTS; i++) {
			WanderAgents[i].target.x += (float) (rand() % 11) - 5;
			WanderAgents[i].target.y += (float) (rand() % 11) - 5;
			_2dvecNormalize(&WanderAgents[i].target);
			WanderAgents[i].target.x *= WanderAgents[i].radius;
			WanderAgents[i].target.y *= WanderAgents[i].radius;
			
			totarget.x = WanderAgents[i].target.x + (Agents[i].heading.x * WanderAgents[i].distance);
			totarget.y = WanderAgents[i].target.y + (Agents[i].heading.y * WanderAgents[i].distance);
			
			Agents[i].position.x += totarget.x * WanderAgents[i].speed;
			Agents[i].position.y += totarget.y * WanderAgents[i].speed;
			
			if (Agents[i].position.x > WIDTH/2)		Agents[i].position.x = - (WIDTH/2);
			if (Agents[i].position.x < - (WIDTH/2))	Agents[i].position.x = WIDTH/2;
			if (Agents[i].position.y < - (HEIGHT/2))	Agents[i].position.y = HEIGHT/2;
			if (Agents[i].position.y > HEIGHT/2)		Agents[i].position.y = - (HEIGHT/2);
			
			_2dvecNormalize(&totarget);
			angle = _2dvecSign(&Agents[i].heading, &totarget) * acos(_2dvecDot(&Agents[i].heading, &totarget) );
			_fabs(angle) > 0.01 ? (Agents[i].rotation = angle) : (Agents[i].rotation = 0.0);
			
			nooverlap(i);	
			draw_agent(&Agents[i]);				
		}
		hpg_flip(); sys_waitRTCTicks(8);
		if(keyb_isLeft()) while(!keyb_isDown());
		
		if(keyb_isRight()) return 0;
		if(keyb_isON()) return 1;
		}
}

int pathfollowing(void)
{
	int i, pointsnumber, index;  
	_2dvector pathpoints[MAXPATHPOINTS + 5], totarget;
	float angle = 0.0;
	char buffer[5];
	
	for(;;) {
		pointsnumber = (rand() % (MAXPATHPOINTS + 1)) + 5;
		for(i = 0; i < pointsnumber; i++) {
			pathpoints[i].x = (float) (rand() % 110) - 50;
			pathpoints[i].y = (float) (rand() % 70) - 34;
		}
		index = 0;
	
		for(;;) {
			hpg_clear();
		
			for(i = 0; i < pointsnumber; i++) {	
				sprintf (buffer, "%d", i + 1);
				hpg_draw_text(buffer, (int) (screencenter.x + pathpoints[i].x), (int) (screencenter.y - pathpoints[i].y));
			}	
			draw_agent(&Agents[0]);
			
			totarget.x = pathpoints[index].x - Agents[0].position.x;
			totarget.y = pathpoints[index].y - Agents[0].position.y;
			
			float distance = sqrt( (totarget.x * totarget.x) + (totarget.y * totarget.y) );
			
			_2dvecNormalize(&totarget);
			angle = _2dvecSign(&Agents[0].heading, &totarget) * acos(_2dvecDot(&Agents[0].heading, &totarget) );
			_fabs(angle) > 0.01 ? (Agents[0].rotation = angle/2.0) : (Agents[0].rotation = 0.0);
				
			Agents[0].position.x += Agents[0].heading.x * 4.0;
			Agents[0].position.y += Agents[0].heading.y * 4.0;

			if(distance < 6.473 )  
				if(++index == pointsnumber) break;	
			
			hpg_flip(); sys_waitRTCTicks(8);
			if(keyb_isRight()) return 0;
			if(keyb_isON()) return 1;	
		}
	}
}

void InitialiseSeek(void)
{
	int i;
	for(i = 0; i < SEEKAGENTS; i++) {
		Agents[i].scale.x = 1; Agents[i].scale.y = 1;
		SeekAgents[i].flag = 1;
		SeekAgents[i].target.x = (float) (rand() % 110) - 50;
		SeekAgents[i].target.y = (float) (rand() % 70) - 35;
	}
}

void InitialiseWander(void)
{
	int i;
	float theta;
	for(i = 0; i < AGENTS; i++) {
		theta = (float) (rand() % 360) * DEGTORAD;
		WanderAgents[i].speed = ( (float) (rand() % 11) + 10.0 ) / 100.0;
		WanderAgents[i].distance = (float) (rand() % 20) + 5.0;
		WanderAgents[i].radius = (float) (rand() % 10) + 10.0;	
		WanderAgents[i].target.x = WanderAgents[i].radius * cos(theta);
		WanderAgents[i].target.y = WanderAgents[i].radius * sin(theta);
		Agents[i].scale.x = 0.5; Agents[i].scale.y = 0.5;
	}
}

void nooverlap(index)
{
	int i;
	_2dvector distance;
	
	for(i = 0; i < AGENTS; i++) {
		if(i == index) continue;
		
		distance.x = Agents[index].position.x - Agents[i].position.x; 	
		distance.y = Agents[index].position.y - Agents[i].position.y; 
		
		float length = sqrt( (distance.x * distance.x) + (distance.y * distance.y) );
		float overlap = Agents[index].radius + Agents[i].radius - length;
		
		if( overlap >= 0 ) {	
			Agents[index].position.x += ( distance.x / length ) * overlap;
			Agents[index].position.y += ( distance.y / length ) * overlap;			
		}		
	}
}

float _2dvecDot(_2dvector *v1, _2dvector *v2)
{
	return (v1->x * v2->x) + (v1->y * v2->y);
}

void _2dvecNormalize(_2dvector *v)
{
	float length = sqrt( (v->x * v->x) + (v->y * v->y) );
	
	if( length > 0 )
	{ v->x /= length; v->y /= length;}
}

int _2dvecSign(_2dvector *v1, _2dvector *v2)
{
	if (v1->y * v2->x > v1->x * v2->y) return 1;
	else  return -1;
}
	