/*
 * flashcrc, (c) 2015 Christoph Giesselink (c dot giesselink at gmx dot de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#include <windows.h>
#include <crtdbg.h>

#define _KB(n)	((n)*1024*2)

#define BOOT	0x86
#define FS		0x18
#define SYSTEM	0x32
#define ROM		0x0F
#define RAM		0xF0

// CRC calculation
static WORD crc_table[] =
{
	0x0000, 0x1081, 0x2102, 0x3183, 0x4204, 0x5285, 0x6306, 0x7387,
	0x8408, 0x9489, 0xA50A, 0xB58B, 0xC60C, 0xD68D, 0xE70E, 0xF78F
};
static __inline WORD UpCRC(WORD wCrc, BYTE byNib)
{
	return (WORD) ((wCrc>>4)^crc_table[(wCrc^byNib)&0xf]);
}

static __inline DWORD Npack(BYTE *a, UINT s)
{
	DWORD r = 0;

	while (s--) r = (r<<4)|a[s];
	return r;
}

static BOOL VerifyCrc(LPBYTE pbyMem, DWORD dwSize, DWORD dwOffset, DWORD dwLength)
{
	WORD wRefCrc,wCrc = 0;

	// address overflow (data length + 4 nibble CRC)
	if (dwOffset + dwLength + 4 > dwSize)
		return FALSE;

	for (; dwLength > 0; --dwLength)		// update CRC
	{
		wCrc = UpCRC(wCrc,pbyMem[dwOffset++]);
	}

	// read reference CRC
	wRefCrc = (WORD) Npack(pbyMem+dwOffset,4);
	return wRefCrc == wCrc;
}

static __inline BOOL CheckRom(LPBYTE pbyMem, DWORD dwSize)
{
	return VerifyCrc(pbyMem,dwSize,0x20A,Npack(pbyMem+0x20A,5));
}

static __inline BOOL CheckSystem(LPBYTE pbyMem, DWORD dwSize)
{
	return VerifyCrc(pbyMem,dwSize,0x20A,Npack(pbyMem+0x100,5));
}

static __inline BOOL CheckRam(LPBYTE pbyMem, DWORD dwSize)
{
	DWORD dwOffset,dwLength,dwType;
	BOOL  bSucc = TRUE;

	dwOffset = 0x20A;

	// not end of data chain
	while (bSucc && (dwType = Npack(pbyMem+dwOffset,2)) != 0xFF)
	{
		dwOffset += 2;						// skip object marker

		// data length info (5) outside page
		if (!(bSucc = (dwOffset + 5 <= dwSize)))
			break;

		// length of object in chain
		dwLength = Npack(pbyMem+dwOffset,5);

		if (dwType != 0)					// object not deleted
		{
			DWORD dwObjOff,dwObjLen;

			// internal object info (16) outside page
			if (!(bSucc = (dwOffset + 16 <= dwSize)))
				break;

			// variables are saved in backup objects because backup objects
			// contain a name field, the data and the CRC of the object

			// check for =DOBAK prologue
			_ASSERT(Npack(pbyMem+dwOffset+6,5) == 0x02B62);

			dwObjOff = dwOffset + 11;		// skip 5 nibble object length
											// + 1 nibble 5
											// + 5 nibble =DOBAK prologue

			// length of =DOBAK object
			dwObjLen = Npack(pbyMem+dwObjOff,5);

			if (!(bSucc = (dwObjLen >= 4)))	// no CRC (4) info in object
				break;

			// check CRC of =DOBAK object
			bSucc = VerifyCrc(pbyMem,dwSize,dwObjOff,dwObjLen-4);
		}

		dwOffset += dwLength;				// next object in chain

		// next object marker (2) inside page
		bSucc = bSucc && (dwOffset + 2 <= dwSize);
	}
	return bSucc;
}

BOOL CheckFlashPage(LPBYTE pbyMem, DWORD dwSize, DWORD dwPage)
{
	BYTE  byType;
	BOOL  bSucc;

	_ASSERT(dwPage >= 0 && dwPage < 16);

	dwPage *= _KB(128);						// convert page no. to data offset
	if (dwPage + _KB(128) > dwSize)			// page not inside flash chip
		return FALSE;

	pbyMem += dwPage;						// page address
	dwPage = _KB(128);						// page size

	byType = (BYTE) Npack(pbyMem+0x200,2);	// get bank type
	if (byType == BOOT)						// 1st half of page is the boot bank
	{
		pbyMem += _KB(64);					// 2nd half of page
		dwPage = _KB(64);					// page size

		// get bank type
		byType = (BYTE) Npack(pbyMem+0x200,2);
	}

	_ASSERT(dwPage == _KB(64) || dwPage == _KB(128));

	switch (byType)
	{
	case FS:  // FRBankFileSystem
	case ROM: // FRBankRom
		bSucc = CheckRom(pbyMem,dwPage);
		break;
	case RAM: // FRBankRam
		bSucc = CheckRam(pbyMem,dwPage);
		break;
	case SYSTEM: // FRBankSystem
		bSucc = CheckSystem(pbyMem,dwPage);
		break;
	default: // illegal bank identifier
		bSucc = FALSE;
	}
	return bSucc;
}
