/*
 * upd49rom, (c) 2015 Christoph Giesselink (c dot giesselink at gmx dot de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#define WIN32_LEAN_AND_MEAN
#define WIN32_EXTRA_LEAN
#define _CRT_SECURE_NO_DEPRECATE
#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "resource.h"

#define VERSION "3.1"

#define BOOT	0x86
#define FS		0x18
#define SYSTEM	0x32
#define ROM		0x0F
#define RAM		0xF0

#define _KB(n)	((n)*1024*2)

#define FLASH	_KB(2*1024)					// size of emulator ROM (2MB)

#define BLOCKHEADOFF	0x2EB4				// offset of block header inside VGER0 file

LPBYTE pVger0 = NULL;

HANDLE hFileFlash = NULL;
HANDLE hMapFlash = NULL;
LPBYTE pFlash = NULL;

HANDLE hFileEmu = NULL;
HANDLE hMapEmu = NULL;
LPBYTE pEmu = NULL;

BOOL   bHeader = FALSE;
BOOL   bRomPack = FALSE;

DWORD  dwFileSize;							// original file size
DWORD  dwVger0Size;							// VGER0 resource size

extern BOOL CheckFlashPage(LPBYTE pbyMem, DWORD dwSize, DWORD dwPage);

VOID LoadVger0(VOID)
{
	HRSRC hResVger0 = FindResource(NULL,MAKEINTRESOURCE(IDR_VGER0),RT_RCDATA);

	// load Vger0 resource
	pVger0 = (LPBYTE) LockResource(LoadResource(NULL,hResVger0));
	dwVger0Size = SizeofResource(NULL,hResVger0);

	// check if block header inside Vger0 resource
	assert(BLOCKHEADOFF + 0x100 < dwVger0Size);

	// check if block header offset is correct
	assert(*(DWORD *) &pVger0[BLOCKHEADOFF] == 0x600d0c26);
	return;
}

BOOL OpenFlash(LPSTR lpszFile)
{
	hFileFlash = CreateFile(lpszFile,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
	if (hFileFlash == INVALID_HANDLE_VALUE)
	{
		hFileFlash = NULL;
		printf("Cannot open file %s.\n", lpszFile);
		return TRUE;
	}
	hMapFlash = CreateFileMapping(hFileFlash, NULL, PAGE_READONLY, 0, 0, NULL);
	if (hMapFlash == NULL)
	{
		printf("CreateFileMapping %s failed.\n", lpszFile);
		return TRUE;
	}
	pFlash = (LPBYTE) MapViewOfFile(hMapFlash, FILE_MAP_READ, 0, 0, 0);
	if (pFlash == NULL)
	{
		printf("MapViewOfFile %s failed.\n", lpszFile);
		return TRUE;
	}
	if (strncmp(pFlash,"HPROM",5))			// not a Flash file
	{
		printf("Invalid flash file %s.\n", lpszFile);
		return TRUE;
	}
	return FALSE;
}

VOID CloseFlash(VOID)
{
	if (pFlash)     UnmapViewOfFile(pFlash);
	if (hMapFlash)  CloseHandle(hMapFlash);
	if (hFileFlash) CloseHandle(hFileFlash);
	return;
}

BOOL OpenEmuRom(LPSTR lpszFile)
{
	hFileEmu = CreateFile(lpszFile,GENERIC_READ|GENERIC_WRITE,0,NULL,OPEN_ALWAYS,0,NULL);
	if (hFileEmu == INVALID_HANDLE_VALUE)
	{
		hFileEmu = NULL;
		printf("Cannot open file %s.\n", lpszFile);
		return TRUE;
	}
	dwFileSize = GetFileSize(hFileEmu,NULL); // actual filesize
	hMapEmu = CreateFileMapping(hFileEmu, NULL, PAGE_READWRITE, 0, FLASH, NULL);
	if (hMapEmu == NULL)
	{
		printf("CreateFileMapping %s failed.\n", lpszFile);
		return TRUE;
	}
	pEmu = (LPBYTE) MapViewOfFile(hMapEmu, FILE_MAP_WRITE, 0, 0, 0);
	if (pEmu == NULL)
	{
		printf("MapViewOfFile %s failed.\n", lpszFile);
		return TRUE;
	}
	return FALSE;
}

VOID CloseEmuRom(VOID)
{
	if (pEmu)     UnmapViewOfFile(pEmu);
	if (hMapEmu)  CloseHandle(hMapEmu);
	if (hFileEmu)
	{
		if (bRomPack)						// rom is packed
		{
			SetFilePointer(hFileEmu,FLASH / 2,NULL,FILE_BEGIN);
			SetEndOfFile(hFileEmu);			// cut the rest
		}
		CloseHandle(hFileEmu);
	}
	return;
}

BOOL PrepareBootBlock(VOID)
{
	BOOL bBootError;
	INT i;

	LPBYTE pSrc = pEmu;
	BOOL bUnpack = FALSE;

	if (dwFileSize == 0)					// no target file
	{
		puts("Creating Boot block ...");
		pSrc = pVger0;						// using resource data
		dwFileSize = dwVger0Size;
	}

	// check for unpacking
	for (i = 0; !bUnpack && i < (INT) dwFileSize; ++i)
		if ((pSrc[i] & 0xF0) != 0)			// higher nibble set
			bUnpack = TRUE;

	if (bUnpack)							// unpack rom image
	{
		puts("Unpacking ROM image ...");

		for (i = dwFileSize - 1; i >= 0; --i)
		{
			pEmu[2*i+1] = pSrc[i] >> 4;
			pEmu[2*i]   = pSrc[i] & 0x0F;
		}
		dwFileSize *= 2;
	}

	bHeader |= (dwFileSize <= _KB(64));		// write new header when only boot block

	// preset rest of ROM
	FillMemory(&pEmu[dwFileSize],FLASH-dwFileSize,0x0F);

	// check bank type
	if (bBootError = (pEmu[0x200] != (BOOT & 0xF) || pEmu[0x201] != (BOOT >> 4)))
		puts("Invalid Boot block.");

	return bBootError;
}

VOID UpdateBlockHeader(VOID)
{
	const DWORD dwPatch[] =
	{
		0x020000, 0x080000, 0x0C0000, 0x100000, 0x140000, 0x180000, 0x1C0000, 0x200000,
		0x240000, 0x280000, 0x2C0000, 0x300000, 0x340000, 0x380000, 0x3C0000
	};

	DWORD dwAddr;
	int   i,j;

	puts("Writing block headers ...");
	for (i = 0; i < sizeof(dwPatch) / sizeof(dwPatch[0]); ++i)
	{
		dwAddr = dwPatch[i];				// nibble patch address

		for (j = BLOCKHEADOFF; j < BLOCKHEADOFF + 0x100; ++j)
		{
			pEmu[dwAddr++] = pVger0[j] & 0xF;
			pEmu[dwAddr++] = pVger0[j] >> 4;
		}
	}

	// write User area header
	dwAddr = 0x20000;
	for (i=0; i < 16; ++i)					// 16 128KB banks
	{
		pEmu[dwAddr+0x200] = RAM & 0xF;
		pEmu[dwAddr+0x201] = RAM >> 4;
		pEmu[dwAddr+0x202] = (BYTE) (i & 0xF);
		pEmu[dwAddr+0x203] = 0x0;
		pEmu[dwAddr+0x204] = 0x1;			// number of erases
		pEmu[dwAddr+0x205] = 0x0;
		pEmu[dwAddr+0x206] = 0x0;
		pEmu[dwAddr+0x207] = 0x0;
		pEmu[dwAddr+0x208] = 0x0;
		pEmu[dwAddr+0x209] = 0x0;

		dwAddr += (i == 0) ? 0x20000 : 0x40000;
	}
	return;
}

VOID InsertFlashData(VOID)
{
	char  buffer[256];
	BYTE  byType;
	BYTE  byBlockNo;
	DWORD dwBlockLen;
	DWORD dwFlashAdr;
	DWORD dwEmuAdr;
	DWORD dwOffset;
	DWORD dwCrcOffset;
	DWORD i,j,dwSize;

	// pFlash[5], ignore number of file parts, use file size

	dwFlashAdr = 0x6;						// index to first name

	dwSize = GetFileSize(hFileFlash, &i);	// size of flash file

	while (dwFlashAdr < dwSize)
	{
		i = pFlash[dwFlashAdr++];			// len of text

		// view module name
		strncpy(buffer,pFlash+dwFlashAdr,i);
		buffer[i] = 0;
		printf("Updating %s ...\n",buffer);

		do
		{
			dwBlockLen = _KB(128);			// normal block has 128KB

			// boot loader
			if (strcmp(buffer,"Part0") == 0)
			{
				byType = BOOT;
				dwOffset = 0x20A;			// normal block begin address
				byBlockNo = 0;				// boot has block no. 0
				dwBlockLen = _KB(64);		// boot block has only 64KB
				dwEmuAdr = 0x00000;			// nibble address of emu file
				break;
			}

			// File System
			if (strcmp(buffer,"FS") == 0)
			{
				byType = FS;
				dwOffset = 0x210;			// data begin at even address before bank ID
				byBlockNo = 0;				// FS has block no. 0
				dwBlockLen = _KB(64);		// rest of boot block has only 64KB
				dwEmuAdr = 0x20000;			// nibble address of emu file
				break;
			}

			// operating system
			if (strcmp(buffer,"System") == 0)
			{
				byType = SYSTEM;
				dwOffset = 0x000;			// system block
				byBlockNo = 1;				// system has block no. 1
				dwEmuAdr = 0x40000;			// nibble address of emu file
				break;
			}

			byType = ROM;
			dwOffset = 0x20A;				// normal block begin address
			// get Part number [1,2,3,4,6,7] -> [2,3,4,5,6,7]
			byBlockNo = pFlash[dwFlashAdr+i-1] - '0';
			if (byBlockNo < 6) ++byBlockNo;
			dwEmuAdr = byBlockNo * 0x40000;	// nibble address of emu file
		}
		while(0);

		dwFlashAdr += i;					// skip text, position of data length

		// fill rest of Bank with 0xF
		for (i = dwOffset; i < dwBlockLen; ++i)
			pEmu[dwEmuAdr+i] = 0xF;

		i = *(DWORD *)(pFlash+dwFlashAdr);	// data length
		dwFlashAdr += sizeof(DWORD);		// skip data length, position of data

		dwCrcOffset = (i - 1) * 2;			// offset to block CRC
		j = (~i+1) & 0x7F;					// rest of 128 byte block

		// copy data
		for (; i > 0; --i, ++dwFlashAdr)
		{
			if (dwOffset >= dwBlockLen)		// block too long
				continue;

			pEmu[dwEmuAdr+dwOffset] = pFlash[dwFlashAdr] & 0xF;
			++dwOffset;
			pEmu[dwEmuAdr+dwOffset] = pFlash[dwFlashAdr] >> 4;
			++dwOffset;
		}

		// fill rest of 128 byte block with 0x0
		for (j *= 2; j > 0; --j)
		{
			if (dwOffset >= dwBlockLen)		// block too long
				continue;

			pEmu[dwEmuAdr+dwOffset] = 0x0;
			++dwOffset;
		}

		// save bank type
		pEmu[dwEmuAdr+0x200] = byType & 0xF;
		pEmu[dwEmuAdr+0x201] = byType >> 4;

		if (byType == SYSTEM)				// restore "System" block default settings
		{
			pEmu[dwEmuAdr+0x202] = byBlockNo;
			pEmu[dwEmuAdr+0x203] = 0x0;
			pEmu[dwEmuAdr+0x204] = 0x1;		// no. of erases
			pEmu[dwEmuAdr+0x205] = 0x0;
			pEmu[dwEmuAdr+0x206] = 0x0;
			pEmu[dwEmuAdr+0x207] = 0x0;
			pEmu[dwEmuAdr+0x208] = 0x0;
			pEmu[dwEmuAdr+0x209] = 0x0;
		}

		if(byType == FS)					// insert CRC offset position
		{
			pEmu[dwEmuAdr+0x20A] = (BYTE) dwCrcOffset & 0xF; dwCrcOffset >>= 4;
			pEmu[dwEmuAdr+0x20B] = (BYTE) dwCrcOffset & 0xF; dwCrcOffset >>= 4;
			pEmu[dwEmuAdr+0x20C] = (BYTE) dwCrcOffset & 0xF; dwCrcOffset >>= 4;
			pEmu[dwEmuAdr+0x20D] = (BYTE) dwCrcOffset & 0xF; dwCrcOffset >>= 4;
			pEmu[dwEmuAdr+0x20E] = (BYTE) dwCrcOffset & 0xF;

			pEmu[dwEmuAdr+0x20F] = 0x0;		// bank revision
			pEmu[dwEmuAdr+0x210] = 0x0;
		}
	}
	return;
}

VOID CheckROMCrc(VOID)
{
	DWORD dwPage;
	CHAR cBuffer[64];
	UINT nPos = 0;

	BOOL bSuccAll = TRUE;

	// check CRC of all pages
	printf("Verifying CRC ...");
	for (dwPage = 0; dwPage < FLASH / _KB(128); ++dwPage)
	{
		// verify CRC(s) of page
		BOOL bSuccCrc = CheckFlashPage(pEmu,FLASH,dwPage);
		bSuccAll = bSuccAll && bSuccCrc;
		nPos += wsprintf(&cBuffer[nPos]," %u%c",dwPage,bSuccCrc ? 'G' : 'B');
	}
	assert(nPos < sizeof(cBuffer) / sizeof(cBuffer[0]));
	puts(bSuccAll ? " ok" : cBuffer);
	return;
}

VOID PackROM(VOID)
{
	LPBYTE pbySrc,pbyDest;
	DWORD  i;

	puts("Packing ROM image ...");

	// pack data in page
	pbySrc = pbyDest = pEmu;
	for (i = 0; i < FLASH / 2; i++)
	{
		*pbyDest =  *pbySrc++;
		*pbyDest |= *pbySrc++ << 4;
		pbyDest++;
	}
	return;
}

VOID UpdateFileDate(VOID)
{
	SYSTEMTIME sSysTime;
	FILETIME   sFileTime;

	GetSystemTime(&sSysTime);
	SystemTimeToFileTime(&sSysTime,&sFileTime);
	SetFileTime(hFileEmu,NULL,NULL,&sFileTime);
	return;
}

UINT main(int argc, char *argv[])
{
	BOOL bErr;
	int  iArg = 1;							// first argument

	printf("HP49G Flash ROM Updater V" VERSION "\n");
	if (argc < 3 || argc > 5)
	{
		printf("\nUsage:\n\t%s [-f -p] <FlashFile> <EmulatorFile>\n", argv[0]);
		return 1;
	}

	while (*argv[iArg] == '-')				// an option
	{
		if (strcmp(argv[iArg],"-f") == 0)	// option "-f", force writing new headers
		{
			bHeader = TRUE;					// write new header
		}

		if (strcmp(argv[iArg],"-p") == 0)	// option "-p", pack result ROM image
		{
			bRomPack = TRUE;				// pack ROM image
		}

		++iArg;								// first file argument
	}

	LoadVger0();							// load Vger0 resource (HP49G boot block)

	bErr = OpenFlash(argv[iArg]);			// open flash file
	if (bErr == FALSE)
		bErr = OpenEmuRom(argv[iArg+1]);	// open emulator file
	if (bErr == FALSE)
		bErr = PrepareBootBlock();			// check boot block

	if (bErr == FALSE)						// no error
	{
		// write new header only on complete OS file
		bHeader = bHeader && (pFlash[5] >= 7);

		if (bHeader) UpdateBlockHeader();	// write new headers
		InsertFlashData();					// make changes
		CheckROMCrc();						// check the ROM CRC's
		if (bRomPack) PackROM();			// pack final ROM image
		UpdateFileDate();					// write new emulator ROM filedate
		puts("Update successful.");
	}
	else									// error
	{
		bRomPack = FALSE;					// leave EmuRom untouched
	}

	CloseFlash();
	CloseEmuRom();

	return bErr;
}
