*****************************************************************************

TITLE QVSOP - Quick VSOP (Jupiter..Neptune 1998-2025) for astro programs

INCLUDE QVXDec.h

*****************************************************************************
* Module name:  QVSOP
* Module type:  Library 1611
* Dest.Comp.:   HP48, HP49
* Language:     System RPL
* Author:       Georg ZOTTI 
* Based on:     QVSOP by Keith FARMER (User RPL implementation)
* Interface:    -> QVXDec.h
* Description:  Quick VSOP positions of Jupiter, Saturn, Uranus, Neptune
*               for 1998...2025 from chapter 32 of
*                   Jean Meeus 
*                   Astronomical Algorithms 2nd ed.
*                   Willmann-Bell, 1998
*                                     
* Edition History:
*  0.0  2000-01-15 KF User RPL directory with data arrays and some programs 
*                     Prepared for on-board library creation (HP49; LibID 1500)
*  0.1  2000-01-16 GZ First SysRPL version, used constants w/o check. Library 1611
*  0.2  2000-01-16 GZ Some internal changes. 
*  0.3  2000-01-17 GZ L mod 360 in QV_D.Y.n>LBR; reordered&renamed xNAMEs
*  0.4  2000-01-20 GZ Added 29 dummy NULLNAMEs to make it expandable.
*                     Put in err.QVSOP
*                     -> Must recompile AARes (-> V2.1.2)!
*  1.0  2000-01-22 GZ removed 16 of the dummy NULLNAMEs (was overkill)
*                     First release version, with AARes 2.1.3.
*  1.01 2000-01-30 GZ Checked constants. Sigh... Fixed 8 numerical errors.
*****************************************************************************

** USER DOCUMENTATION (This comes into the manual)
*" This ``Quick VSOP'' library provides faster planetary positions for the years
*" 1998\ldots2025. 
*"
*" \details{%
*" This library was developed with the help of \name{Keith Farmer}. If
*" installed, it will speed up calculations of the positions of the gas planets
*" for \AARes. It is not required, should you have too little memory available.
*"
*" It does not autoattach, because it is mainly used by \AARes. You might
*" attach it manually (type \texthp{1611 ATTACH}) should you need the provided
*" commands. 
*"
*" \QVSOP{} is independent of the rest of \Urania, and is provided in source
*" code and libraries for both the HP48 and HP49. 

**" See the \file{QVSOP.zip} in the package.  
*" }
**************************************************************************
** DEVELOPER INFORMATION:
**
** WARNING! DO NOT CHANGE THE ORDER OF THE FIRST 17 xNAMEs and NULLNAMEs!
** First are the user RPL entries (xNAMEs). The NULLNAMEs named QV_dummyXX
** may be _replaced_ by xNAMEs, but MUST NOT BE DELETED! The NULLNAMEs
** starting at QV_D.Y.n>LBR must stay at the current ROMPTR
** position. After those ROMPTRs follow the data arrays. After the
** data arrays, you may put anything you like.
** 
** This means: replace QV_dummyXX by new user commands (xNAMEs), put new
** NULLNAMEs after the data arrays, and don't change the order of the
** commands included here, or build your own library and change
** Library ID! 
**

** If you don't keep these rules, software that uses QVSOP, like Urania/AARes
** V2.1+ or StarMaHP V3.1+ (?) will almost certainly crash!
**

** AARes 2.1.2   depends on QV_D.Y.n>LBR being ROMPTR #64B003. (Tested concept)
** AARes 2.1.3+  depends on QV_D.Y.n>LBR being ROMPTR #64B010. 


xROMID        64B
ASSEMBLE
QVROMID EQU  #64B

RPL
#IF HP49
FEXTERNAL XEQ>ARRY
#ENDIF

*****************************************************************************
* QVcfg
* Interface :   ( --> )
* Description : Attach this library at system warmstart
*               NO! No config object. keep lib as hidden as possible.
*****************************************************************************
*
*=QVcfg
*RPL
**:: DOBINT QVROMID XEQSETLIB ;          ( *Autoattach* )
* NOP
****************************************************************************
*" \begin{Ucommands}
****************************************************************************
*" \UCMD{QVSOP}{Calculates L, B, R for all giant planets} 
*" {
*" \Stack{ 2: DD.dd & 2: [$L_5$ $B_5$ $R_5$ $L_6$ $B_6$ $R_6$ 
*"                        $L_7$ $B_7$ $R_7$ $L_8$ $B_8$ $R_8$]\\
*"         1: YYYY  & 1: 1  }
*" or
*" \Stack{          & 3: DD.dd \\
*"         2: DD.dd & 2: YYYY \\
*"         1: YYYY  & 1: 0  }
*" $YYYY$ must be in the range given by \ucmd{range.QVSOP} to give a 
*" ``good'' result. $L_n$ may be greater than $360\deg$!}
****************************************************************************
ASSEMBLE
        CON(1)   8
RPL
xNAME QVSOP
:: CK2&Dispatch REALREAL 
:: 
   QV_D.Y>all COERCEFLAG

 ; ( End_REALREAL )
; ( End_QVSOP )

****************************************************************************
*" \UCMD{rng.QVSOP}{Returns info about range of years in this \QVSOP{} lib} 
*" {
*" \Stack{& 2: first year\\
*"        & 1: last year  }
*" }
* CHANGE HERE IF YOU REMOVE DATA!
****************************************************************************
ASSEMBLE
        CON(1)   8
RPL
xNAME rng.QVSOP
:: CK0 % 1998 % 2025 ;

*  was: \<< HASH DUP HEAD SWAP REVLIST TAIL HEAD \>>
****************************************************************************
*" \UCMD{err.QVSOP}{Returns vector of maximum error} 
*" { This may be useful to estimate accuracy of the results.
*" \Stack{& 1: [[$\Delta L_5$ $\Delta B_5$ $\Delta R_5$] 
*"              [$\Delta L_6$ \ldots] \ldots [\ldots $\Delta R_8$]]  }
*" Values are given in degrees. A value of $-1$ means ``unknown'' }
* V0.4 new; constant vector. -1 means unknown
****************************************************************************
ASSEMBLE
        CON(1)   8
RPL
xNAME err.QVSOP
:: ARRY 4 3 [ 1.38888888889E-5 -1.0 -1.0 
              5.55555555556E-6 -1.0 -1.0 
              5.55555555556E-6 -1.0 -1.0 
              5.55555555556E-6 -1.0 -1.0 ]
 ;
****************************************************************************
*" \UCMD{src.QVSOP}{Returns info about \QVSOP{} source} 
*" {
*" \Stack{& 1: info string  }
*" }
****************************************************************************
ASSEMBLE
        CON(1)   8
RPL
xNAME src.QVSOP
 :: $ "Meeus, Jean\0AAstronomical Algorithms, \0A2nd ed. 1998" ;

****************************************************************************
** Here are some placeholders for future User RPL commands 
** (Put e.g. CON(1) 8 in front and change to xNAME mycmd.QVSOP)
****************************************************************************
NULLNAME QV_dummy04 
:: ;
NULLNAME QV_dummy05 
:: ;
NULLNAME QV_dummy06 
:: ;
NULLNAME QV_dummy07 
:: ;
NULLNAME QV_dummy08 
:: ;
NULLNAME QV_dummy09 
:: ;
NULLNAME QV_dummy0A 
:: ;
NULLNAME QV_dummy0B 
:: ;
NULLNAME QV_dummy0C 
:: ;
NULLNAME QV_dummy0D 
:: ;
NULLNAME QV_dummy0E 
:: ;
NULLNAME QV_dummy0F 
:: ;
*" \end{Ucommands}
****************************************************************************
****************************************************************************
* Internal commands below.
*" \begin{Scommands}
****************************************************************************
*" \SCMD{QV\_D.Y.n>LBR}{Calculates L, B, R for given giant planet} 
*" {
*" \Stack{              & 4: $L$\\
*"         3: DD.dd     & 3: $B$\\
*"         2: YYYY      & 2: $R$\\
*"         1: \%n[5..8] & 1: TRUE}
*" or \Stack{              & 4: DD.dd\\
*"            3: DD.dd     & 3: YYYY\\
*"            2: YYYY      & 2: \%n\\
*"            1: \%n[5..8] & 1: FALSE}


*" YYYY must be 1998\ldots2025, n 5\ldots8 for a ``TRUE'' result!
*" $L$ is always $[0\ldots360]$.
*" }
****************************************************************************
NULLNAME QV_D.Y.n>LBR
:: 

DUP %5 %>= OVER %8 %<= ANDITE ( good planet? )
::
  UNROT QV_D.Y>all
* %n [] T    or   %n DDD.ddd YYYY F 
  ITE
      ::
* %n arry
        SWAP COERCE #5- #3 #* #1+ ( 5->1, ... 8->10, i.e. start indices for L/B/R_planet )
* arry index
        DUPUNROT PULLREALEL %360 %MOD
*  index arry  L
        UNROTOVER #1+ PULLREALEL
* L index arry B
        UNROTSWAP #2+ PULLREALEL SWAPDROP
        TRUE
      ;
      :: ROT FALSE ; ( ITE_Else ) 
; 
FALSE ( ITE_Else, bad planet )

; ( END_QV_D.Y.n>LBR )
****************************************************************************
*" \SCMD{QV\_D.Y>all}{Calculates L, B, R for all giant planets} 
*" {
*" \Stack{ 2: DDD.dd & 2: [L5 B5 R5 L6 B6 R6 L7 B7 R7 L8 B8 R8] \\
*"         1: YYYY   & 1: TRUE }
*"  or
*" \Stack{           & 3: DDD.dd\\
*"         2: DDD.dd & 2: YYYY \\
*"         1: YYYY   & 1: FALSE} 
*" if YYYY outside range (here 1998\ldots2025)!
*" }
** CHANGE HERE ( **** ) AND THE ARRY IF YOU REMOVE DATA!
****************************************************************************
NULLNAME QV_D.Y>all
:: DUP % 1998 %>= OVER % 2025 %<= ANDITE   ( **** ) 
 ::   COERCE 1997 #-                       ( **** )
   ARRY [ QV_P1998 QV_P1999 QV_P2000 QV_P2001 QV_P2002 QV_P2003 QV_P2004 
          QV_P2005 QV_P2006 QV_P2007 QV_P2008 QV_P2009 QV_P2010 QV_P2011 
          QV_P2012 QV_P2013 QV_P2014 QV_P2015 QV_P2016 QV_P2017 QV_P2018 
          QV_P2019 QV_P2020 QV_P2021 QV_P2022 QV_P2023 QV_P2024 QV_P2025 ]
   GETATELN DROP EVAL ( get that array )

* DD.dd [[Array_of_YYYY]]

**       1. T 365. / DUP SQ DUP2 *
**      DUP2 OVER SQ UNROT * { 6 1 } \->ARRY * 1.

   %1 ROT % 365 %/ 
* [[data]] %1 %t 
DUPDUP %* 2DUP %*
* [[data]] %1 %t %t^2 %t^3
OVERDUP %* 3PICK3PICK %*   
* [[data]] %1 %t %t^2 %t^3 %t^4 %t^5
{ %6 %1 } 
#IF HP49 XEQ>ARRY
#ELSE    XEQ>ARRAY
#ENDIF 
        ' x* EvalNoCK
 TRUE ;
 FALSE ( ITE_outOfTime )

; ( End_QV_D.Y>all )

** old source
*QVSOP @ D.dd Y -> [L5 B5 R5 L6 B6 R6 L7 B7 R7 L8 B8 R8] 1.
*      @ or        D.dd Y 0.
*\<< \-> T Y
*  \<<
*      IF HASH Y POS DUP 0. ==
*      THEN DROP T Y 0.
*      ELSE HASH SWAP 1. + GET EVAL 1. T 365. / DUP SQ DUP2 *
*           DUP2 OVER SQ UNROT * { 6 1 } \->ARRY * 1.
*      END
*    \>>
*  \>>


****************************************************************************
*" \SCMD{QV_P$yyyy$}{data arrays} 
*" { $yyyy$ is 1998\ldots2025. Internal use only. 
*" \Stack{& 1: array 12x6}
*" }
****************************************************************************


NULLNAME QV_P1998
ARRY 12 6
[ 329.6821899 32.5373311  .6251361 -.0898353 -.0386157  .0083980 
    -.9881379  -.4836213  .1496847  .0343504 -.0042712 -.0008253 
    5.0212516  -.0962003  .0267709  .0046167  .0001739 -.0003582 
   19.7448573 12.6969726  .1368882  .0062460 -.0099932  .0034523 
   -2.4797134  -.0370230  .0601488  .0020497 -.0007432  .0001794 
    9.3669872  -.1015298  .0037320  .0007303 -.0002364  .0001522 
  308.3946411  4.0234022 -.0099074  .0013007 -.0021907  .0010334 
    -.6281267  -.0316524  .0015991  .0001139 -.0000869  .0000281 
   19.8439293   .0443881 -.0017243 -.0013796  .0013722 -.0004733 
  299.5407539  2.1902131  .0002159 -.0000101 -.0007024  .0004022 
     .3746913  -.0657516 -.0003038  .0001425 -.0001276  .0000450 
   30.1452426  -.0114799  .0000256 -.0016443  .0017726 -.0006393 ]


NULLNAME QV_P1999
* 2000-01-28 Saturn R-A1 had wrong sign!
ARRY 12 6
[   2.7246041 33.4053704  .2040247 -.1756398 -.0005659  .0000648 
   -1.2928206  -.1024089  .2189535  .0088476 -.0085174  .0000270 
    4.9562546  -.0298757  .0376516  .0032669 -.0026228  .0005117 
   32.5784232 12.9666139  .1294965 -.0105762  .0076613 -.0036652 
   -2.4551018   .0873406  .0635537  .0003573 -.0001129 -.0001191 
    9.2698355  -.0920465  .0052931  .0019792 -.0013122  .0004223 
  312.4082792  4.0039196 -.0113243  .0039059 -.0036098  .0011560 
    -.6581250  -.0283220  .0017436 -.0001090  .0001411 -.0000552 
   19.8861125   .0399427 -.0024498  .0011943 -.0016195  .0007214 
  301.7308726  2.1898447 -.0014772  .0025844 -.0027128  .0009371 
     .3086958  -.0662200 -.0001651 -.0001133  .0001518 -.0000615 
   30.1332773  -.0124411 -.0005349  .0010464 -.0014170   .0006547 ]


NULLNAME QV_P2000
ARRY 12 6
[  36.1578582 33.2848762 -.3202376 -.1607197  .0101019  .0048032 
   -1.1759189   .3281328  .1949128 -.0233461 -.0076376  .0015479 
    4.9651863   .0472665  .0371120 -.0037156 -.0001438 -.0002643 
   45.6679535 13.2064010  .1101519 -.0047180 -.0030779  .0012521 
   -2.3040822   .2144789  .0629352 -.0008924 -.0005166  .0000635 
    9.1841714  -.0786804  .0080428 -.0004246  .0011535 -.0004310 
  316.4023267  3.9842795 -.0081955 -.0029436  .0035933 -.0014074 
    -.6847265  -.0248706  .0017191  .0001005 -.0001187  .0000503 
   19.9239016   .0357276 -.0020273 -.0000038  .0001878 -.0001152 
  303.9200489  2.1884417  .0000758 -.0019024  .0022339 -.0008929 
     .2422877  -.0665872 -.0001798  .0000780 -.0000946  .0000426 
   30.1205855  -.0128018  .0000027 -.0000216  .0002692 -.0001503 ]

NULLNAME QV_P2001
ARRY 12 6
[  69.0649681 32.2226338 -.6998931 -.0829214  .0310692 -.0040156 
    -.6805959   .6256131  .0938187 -.0391518 -.0003911  .0007610 
    5.0457384   .1086124  .0221681 -.0046960 -.0019856  .0006987 
   59.0146928 13.4069581  .0860321 -.0066438 -.0009758 -.0005570
   -2.0270929   .3362430  .0575093 -.0021434 -.0007976  .0001291 
    9.1136636  -.0613166  .0091754  .0023443 -.0018009  .0006689 
  320.3885200  3.9664687 -.0097794  .0049444 -.0054151  .0021564 
    -.7079044  -.0213463  .0017552  .0000082  .0000180 -.0000143 
   19.9577580   .0318173 -.0013643 -.0007121  .0004838 -.0000877 
  306.1139979  2.1874304 -.0010030  .0025367 -.0029947  .0012256 
     .1753634  -.0668803 -.0001554  .0000482 -.0000188  .0000002 
   30.1078496  -.0125395  .0008452 -.0009250  .0007678 -.0001989 ]

NULLNAME QV_P2002
ARRY 12 6
[ 100.5318411 30.6783457 -.7959767  .0046134  .0241846 -.0027664 
     .0000541   .6980207 -.0179458 -.0339429  .0040248 -.0001891 
    5.1705360   .1343765  .0032467 -.0076620  .0016344 -.0005326 
   72.4995064 13.5523675  .0590572 -.0190902  .0072511 -.0026972 
   -1.6361526   .4422784  .0477640 -.0045641  .0001316 -.0001680
    9.0627346  -.0398191  .0118501  .0000014  .0008616 -.0004634 
  324.3468951  3.9508084 -.0074756  .0006290  .0007218 -.0005844 
    -.7274836  -.0178115  .0018137 -.0000903  .0000868 -.0000288 
   19.9878950   .0284669 -.0021314  .0014865 -.0017320  .0006781 
  308.3011927  2.1871600 -.0001927  .0006350  .0000066 -.0001904 
     .1083572  -.0671220 -.0000564 -.0000988  .0001133 -.0000401 
   30.0957992  -.0115299  .0000961  .0015161 -.0018195  .0007336 ]

NULLNAME QV_P2003
* 2000-01-28: Saturn B-A1 fixed
ARRY 12 6
[ 130.4402417 29.1831608 -.6703649  .0749318  .0072343 -.0006340 
     .6500218   .5754659 -.0977839 -.0188353  .0024263  .0001555 
    5.3015990   .1218037 -.0151896 -.0040044 -.0006998  .0004889 
   86.0963948 13.6289016  .0157981 -.0062851 -.0070380  .0025844 
   -1.1507107   .5238144  .0331432 -.0048639 -.0008078  .0002564 
    9.0351652  -.0149521  .0129003  .0006775 -.0007201  .0002840 
  328.2909943  3.9377576 -.0051693  .0000154  .0000545  .0000916 
    -.7435137  -.0142493  .0017331  .0001269 -.0001433  .0000554 
   20.0146632   .0250860 -.0012394 -.0010780  .0011495 -.0004444 
  310.4886113  2.1877683  .0012570 -.0008609  .0008138 -.0002414 
     .0411532  -.0672759 -.0001019  .0001349 -.0001418  .0000566 
   30.0847956  -.0104405  .0009560 -.0010546  .0011607 -.0004588 ]

NULLNAME QV_P2004
ARRY 12 6
[ 159.0345698 28.0927741 -.4025854  .0863880  .0149846 -.0039962 
    1.1114503   .3338606 -.1383051 -.0085903  .0027221 -.0003263 
    5.4039978   .0790223 -.0266881 -.0038764  .0019314 -.0005949 
   99.7303559 13.6261946 -.0183448 -.0223133  .0115296 -.0045045 
    -.5991684   .5735498  .0161432 -.0061982  .0002143 -.0001050 
    9.0333550   .0114227  .0127670  .0006005 -.0004786  .0000398 
  332.2237441  3.9282198 -.0059965  .0053021 -.0048520  .0016624 
    -.7559908  -.0107012  .0017997 -.0000888  .0001159 -.0000498 
   20.0381369   .0217743 -.0017986  .0007608 -.0012460  .0005746 
  312.6773480  2.1898129 -.0000225  .0035948 -.0037652  .0013826 
    -.0261749  -.0673619 -.0000062 -.0000484  .0000897 -.0000402 
   30.0749585  -.0093289  .0006114  .0002615 -.0007761  .0004204 ]

NULLNAME QV_P2005
ARRY 12 6
[ 186.8977139 27.5863948 -.0972861  .1182676 -.0111922  .0041159 
    1.3009218   .0399162 -.1506563 -.0005039  .0017402  .0000343 
    5.4538435   .0186227 -.0322694 -.0003512  .0000610  .0002636 
  113.3600304 13.5460315 -.0593246 -.0088776 -.0040233  .0022106 
    -.0139545   .5875677 -.0020420 -.0060833  .0001101  .0000440 
    9.0578081   .0371069  .0127598 -.0013820  .0010512 -.0004217 
  336.1588225  3.9209574 -.0018016 -.0026331  .0039271 -.0016943 
    -.7649346  -.0071424  .0017870 -.0000051 -.0000173  .0000138 
   20.0582522   .0183255 -.0020410  .0005349 -.0005401  .0001568 
  314.8743572  2.1923691  .0019779 -.0014836  .0021474 -.0009828 
    -.0937264  -.0673602  .0000461 -.0000126  .0000113  .0000024 
   30.0661238  -.0083386  .0000697  .0007833 -.0009169  .0003301 ]

NULLNAME QV_P2006
ARRY 12 6
[ 214.4980139 27.7220245  .2346710  .0921343  .0115682 -.0055378 
    1.1914523  -.2557633 -.1417826  .0075267  .0009788  .0004777 
    5.4401702  -.0454318 -.0307570  .0012036  .0012608 -.0003290 
  126.8360470 13.3955444 -.0911468 -.0122376  .0049237 -.0020233 
     .5656419   .5658951 -.0193918 -.0050723 -.0000684  .0001223 
    9.1069223   .0606146  .0103141  .0007824 -.0014604  .0005055 
  340.0775781  3.9168193 -.0023459  .0037264 -.0041381  .0016941 
    -.7702986  -.0035834  .0017364  .0000929 -.0000886  .0000294 
   20.0746882   .0144546 -.0014695 -.0010956  .0009112 -.0002909 
  317.0683853  2.1956170  .0016596  .0010143 -.0015197  .0006827 
    -.1610394  -.0672478  .0000391  .0001174 -.0001019  .0000363 
   30.0580514  -.0078999  .0007450 -.0014202  .0012833 -.0004389 ]


NULLNAME QV_P2007
* 2000-01-30: Uranus R-A1, B-A3 fixed
ARRY 12 6
[ 242.5528742 28.4867192  .5229238  .1039235 -.0236677  .0055436 
     .8028896  -.5104669 -.1083669  .0148232  .0037680 -.0003019 
    5.3661167  -.0999313 -.0222721  .0033557  .0007824 -.0000057 
  140.1311075 13.1861124 -.1140389 -.0128927  .0055090 -.0012513 
    1.1071268   .5122260 -.0337146 -.0045624  .0006098 -.0001060 
    9.1776786   .0802443  .0090515 -.0020322  .0016445 -.0007113 
  343.9933339  3.9152077 -.0013000  .0021909 -.0010142  .0000326 
    -.7721119  -.0000415  .0018199 -.0001263  .0001434 -.0000556 
   20.0871980   .0104493 -.0026204  .0015434 -.0019845  .0008009 
  319.2658391  2.1993302  .0008806  .0022165 -.0020477  .0006061 
    -.2281964  -.0670460  .0001690 -.0000905  .0001259 -.0000496 
   30.0503208  -.0077131 -.0003349  .0010779 -.0015939  .0006915 ]

NULLNAME QV_P2008
ARRY 12 6
[ 271.6483166 29.7769198  .7477661  .0397082 -.0020776 -.0067762 
     .2023449  -.6691526 -.0441376  .0280087  .0026931 -.0000486 
    5.2480457  -.1313144 -.0081191  .0061518  .0001181 -.0000020 
  153.1945460 12.9351827 -.1357811  .0009022 -.0048274  .0020824 
    1.5815797   .4330239 -.0447726 -.0028207  .0000966  .0001016 
    9.2658754   .0953134  .0060668 -.0002510 -.0007978  .0003521 
  347.9084509  3.9152947  .0016034 -.0020211  .0024112 -.0009125 
    -.7703719   .0035180  .0017552  .0000880 -.0001148  .0000494 
   20.0953867   .0058701 -.0021983 -.0006627  .0007453 -.0003326 
  321.4668247  2.2025604  .0023563 -.0020028  .0021281 -.0008550
    -.2950876  -.0667215  .0001770  .0000796 -.0000857  .0000387 
   30.0424482  -.0081002 -.0001917 -.0005047  .0005724 -.0002591 ]

NULLNAME QV_P2009
ARRY 12 6
[ 302.2897525 31.3542001  .7873094 -.0101742 -.0408975  .0067920 
    -.4821079  -.6625902  .0554601  .0371385  .0014699 -.0011593 
    5.1145278  -.1285893  .0115895  .0057933  .0005733 -.0002814 
  166.0267815 12.6564216 -.1389283 -.0113369  .0111521 -.0038478 
    1.9681285   .3356251 -.0517770 -.0015747  .0001886  .0000333 
    9.3668471   .1052642  .0034728 -.0009956  .0005031 -.0002889 
  351.8355595  3.9176400 -.0004279  .0051042 -.0052751  .0019119 
    -.7650567   .0070885  .0017676  .0000065  .0000153 -.0000132 
   20.0988108   .0008070 -.0025051  .0000724 -.0005166  .0002948 
  323.6770543  2.2055920  .0003305  .0027771 -.0034717  .0013507 
    -.3617811  -.0662780  .0002330  .0000464 -.0000133 -.0000003 
   30.0339404  -.0090077 -.0001287 -.0006463  .0002930  .0000011 ]

NULLNAME QV_P2010
ARRY 12 6
[ 334.3869822 32.7683636  .5760783 -.1238230 -.0142657 -.0012614 
   -1.0517888  -.4401708  .1643785  .0313220 -.0042798 -.0010832 
    5.0036133  -.0871279  .0290746  .0061802 -.0019050  .0002955 
  178.5402422 12.3700943 -.1442983  .0062540 -.0070508  .0034336 
    2.2506239   .2282659 -.0549763 -.0006504  .0003647 -.0000512 
    9.4748026   .1097941  .0013751 -.0018436  .0010008 -.0003215 
  355.7545126  3.9204849  .0021446 -.0022638  .0034613 -.0016180 
    -.7561921   .0106378  .0018161 -.0000961  .0000892 -.0000301 
   20.0969632  -.0045723 -.0032576  .0012002 -.0012976  .0004622 
  325.8836328  2.2074198  .0005140 -.0004733  .0008905 -.0005519 
    -.4277933  -.0657282  .0003431 -.0000674  .0000842 -.0000292 
   30.0244517  -.0100091 -.0010648  .0015176 -.0017270  .0006758 ]

NULLNAME QV_P2011
ARRY 12 6
[   7.5920741 33.4861111  .1105793 -.1707292 -.0141162  .0086056 
   -1.3016221  -.0399523  .2216422  .0051585 -.0103989  .0009179 
    4.9501306  -.0166052  .0395130  .0001809 -.0004479 -.0002501 
  190.7686750 12.0889793 -.1358846 -.0011339  .0056670 -.0022230 
    2.4235765   .1175682 -.0552441  .0005608 -.0000541  .0000983 
    9.5848075   .1094277 -.0018981  .0002135 -.0009596  .0003480 
  359.6767218  3.9238438  .0014779  .0014628 -.0020231  .0009043 
    -.7437752   .0141902  .0017203  .0001200 -.0001449  .0000558 
   20.0894980  -.0103940 -.0023728 -.0011701  .0012464 -.0004592 
  328.0914319  2.2078669  .0004037 -.0007458  .0003201 -.0000318 
    -.4931907  -.0650512  .0003298  .0001162 -.0001192  .0000476 
   30.0138443  -.0111563  .0000344 -.0012071  .0014028 -.0005328 ]

NULLNAME QV_P2012
ARRY 12 6
[  41.0125248 33.1813654 -.4045897 -.1565926  .0274250 -.0029575 
   -1.1242548   .3817953  .1837984 -.0278906 -.0055891  .0010414 
    4.9725213   .0599571  .0345019 -.0022206 -.0024916  .0006584 
  202.7240798 11.8254081 -.1238258  .0025325  .0002032  .0007134 
    2.4865057   .0090374 -.0530558  .0012504 -.0000236  .0000373 
    9.6919390   .1041370 -.0032098 -.0023756  .0019690 -.0007545 
    3.6023875  3.9276448  .0003642  .0032150 -.0024109  .0006078 
    -.7278338   .0176871  .0017578 -.0000982  .0001124 -.0000490
   20.0763484  -.0159219 -.0030759  .0015680 -.0019150  .0008013 
  330.2992450  2.2076168 -.0016178  .0032781 -.0033868  .0011936 
    -.5578675  -.0642839  .0004196 -.0000338  .0000656 -.0000294 
   30.0023853  -.0117358 -.0002383  .0009535 -.0013463  .0006305 ]

NULLNAME QV_P2013
ARRY 12 6
[  73.7448347 31.9934973 -.7339396 -.0653781  .0248588 -.0004753 
    -.5893218   .6489931  .0771672 -.0397064  .0006613  .0006391 
    5.0632434   .1156997  .0196696 -.0073820  .0009734 -.0003958 
  214.4608626 11.5891127 -.1119922  .0108332 -.0055004  .0020909 
    2.4434955  -.0935032 -.0489102  .0013073  .0003353 -.0000939 
    9.7919646   .0946956 -.0058810  .0000588 -.0009202  .0004417 
    7.5425796  3.9313963  .0034892 -.0034020  .0043364 -.0017360
    -.7083658   .0211235  .0017049 -.0000243 -.0000146  .0000125 
   20.0577472  -.0210600 -.0025105  .0002189  .0000573 -.0000987 
  332.5123744  2.2065963  .0001903 -.0018869  .0025870 -.0011286 
    -.6219032  -.0634276  .0004650 -.0000083  .0000041  .0000041 
   29.9906169  -.0116051 -.0000613  .0007836 -.0006659  .0002076 ]

NULLNAME QV_P2014
ARRY 12 6
[ 104.9633978 30.4264402 -.7914405  .0238249  .0197071 -.0032518 
     .0984324   .6900469 -.0320762 -.0306204  .0027843  .0002163 
    5.1918082   .1348517 -.0007139 -.0050416 -.0011251  .0005752 
  225.9454068 11.3858943 -.0889779 -.0011738  .0074964 -.0025684 
    2.3026308  -.1865194 -.0440229  .0023516 -.0005050  .0002148 
    9.8803594   .0816029 -.0072784 -.0011157  .0009451 -.0004068 
   11.4766635  3.9369705  .0016703  .0045735 -.0046517  .0017806 
    -.6855638   .0244651  .0016003  .0000765 -.0000918  .0000307 
   20.0343542  -.0256833 -.0017734 -.0004195  .0002851 -.0000423 
  334.7187325  2.2061042 -.0004344  .0017465 -.0020850  .0008747 
    -.6848660  -.0624850  .0004570  .0000952 -.0000878  .0000322 
   29.9792758  -.0110295  .0010366 -.0011668  .0011044 -.0003637 ]

NULLNAME QV_P2015
* 2000-01-30: Uranus L-A4, Neptune L-A2, B-A0 fixed
ARRY 12 6
[ 134.6386778 28.9775308 -.6276824  .0669581  .0168200 -.0032219 
     .7287832   .5462341 -.1048855 -.0184840  .0039686 -.0005057 
    5.3203546   .1166386 -.0168448 -.0058563  .0020150 -.0006241 
  237.2460773 11.2217378 -.0741577  .0144744 -.0097920  .0041482 
    2.0741498  -.2684658 -.0379173  .0018377  .0003680 -.0001549 
    9.9541066   .0654485 -.0082795 -.0011954  .0006345 -.0001263 
   15.4170066  3.9442719  .0041430 -.0006173  .0023110 -.0012324 
    -.6594830   .0276789  .0016264 -.0001459  .0001436 -.0000559 
   20.0067208  -.0295335 -.0024078  .0017729 -.0019111  .0007160 
  336.9249386  2.2065131 -.0003495  .0016632 -.0010552  .0001916 
    -.7468544  -.0614776  .0005549 -.0000662  .0000906 -.0000355 
   29.9688569  -.0098315  .0001907  .0015577 -.0019486  .0007998 ]

NULLNAME QV_P2016
ARRY 12 6
[ 163.0690823 27.9743024 -.3634612  .1078701 -.0037752  .0019072 
    1.1551108   .2943723 -.1414807 -.0066130  .0015662  .0001482 
    5.4156829   .0703597 -.0283421 -.0018086 -.0003652  .0004005 
  248.4024880 11.0981886 -.0498654  .0036715  .0045755 -.0019976 
    1.7698174  -.3380817 -.0315546  .0022754 -.0001424  .0000873 
   10.0105884   .0472140 -.0099693  .0005838 -.0008777  .0003398 
   19.3658827  3.9538550  .0059123 -.0002440  .0005465 -.0001220 
    -.6302359   .0307918  .0015029  .0000655 -.0001126  .0000487 
   19.9753573  -.0331294 -.0012957 -.0009381  .0011617 -.0004855 
  339.1319019  2.2075407  .0015261 -.0015291  .0018420 -.0007186 
    -.8077883  -.0603801  .0005615  .0000596 -.0000680  .0000312 
   29.9596250  -.0086143  .0008367 -.0006548  .0008027 -.0003576 ]

NULLNAME QV_P2017
ARRY 12 6
[ 190.8614472 27.5648846 -.0376795  .0950355  .0125018 -.0044412 
    1.3030989  -.0022487 -.1508309  .0009584  .0013231  .0001389 
    5.4559510   .0085834 -.0322787 -.0010422  .0016684 -.0004893 
  259.4872465 11.0177308 -.0281315  .0082957 -.0029458  .0016845 
    1.4013204  -.3946323 -.0249619  .0026334 -.0002708  .0000867 
   10.0479536   .0271211 -.0096701 -.0018104  .0017667 -.0006354 
   23.3366976  3.9666314  .0051813  .0043644 -.0034992  .0010928 
    -.5978470   .0337938  .0014529 -.0000100  .0000120 -.0000117 
   19.9405709  -.0362844 -.0017189  .0010807 -.0014608  .0006325 
  341.3466171  2.2098703  .0001906  .0033414 -.0035370  .0013188 
    -.8677463  -.0591918  .0005982  .0000475 -.0000266  .0000068 
   29.9516174  -.0074637  .0007698 -.0001344 -.0002736  .0002032 ]

NULLNAME QV_P2018
* 2000-01-30: Uranus B-A4 fixed
ARRY 12 6
[ 218.4917484 27.8027013  .2742538  .1168917 -.0176575  .0051412 
    1.1524397  -.2950583 -.1382465  .0070141  .0027396 -.0001175 
    5.4323925  -.0548542 -.0297567  .0018530  .0004233  .0001513 
  270.4838802 10.9829864 -.0084115  .0134246 -.0058676  .0019517 
     .9841755  -.4373125 -.0176187  .0019585  .0004897 -.0001736 
   10.0647254   .0062681 -.0108303  .0008458 -.0013031  .0005968 
   27.3104684  3.9814942  .0094084 -.0035588  .0051513 -.0021644 
    -.5626100   .0366587  .0014472 -.0001155  .0000929 -.0000313 
   19.9028200  -.0391742 -.0017216  .0006973 -.0005305  .0001099 
  343.5578014  2.2126870  .0017473 -.0009114  .0017482 -.0008936 
    -.9263123  -.0579263  .0006869 -.0000502  .0000639 -.0000221 
   29.9447188  -.0063999 -.0000426  .0012655 -.0014809  .0005456 ]

NULLNAME QV_P2019
ARRY 12 6
[ 246.6730789 28.6564988  .5720391  .0728729  .0058592 -.0066096 
     .7287711  -.5401173 -.1020927  .0181279  .0018647  .0004885 
    5.3502092  -.1063772 -.0206549  .0041448  .0007904 -.0001698 
  281.4679638 10.9926030  .0191397  .0019659  .0056652 -.0019957 
     .5315188  -.4655717 -.0106033  .0029422 -.0005030  .0002128 
   10.0603027  -.0151233 -.0105719 -.0006684  .0008559 -.0003573 
   31.3007991  3.9995492  .0087290  .0032347 -.0032874  .0013338 
    -.5245579   .0394244  .0013056  .0001070 -.0001440  .0000560
   19.8622009  -.0420977 -.0009882 -.0008669  .0008068 -.0002803 
  345.7721788  2.2160304  .0019492  .0001302 -.0004086  .0002250
    -.9835601  -.0565564  .0006838  .0000946 -.0000980  .0000402 
   29.9386064  -.0059103  .0006087 -.0011338  .0010820 -.0004058 ]

NULLNAME QV_P2020
ARRY 12 6
[ 275.9737393 30.0100340  .7588199  .0547962 -.0346193  .0058199 
     .1070422  -.6800467 -.0317680  .0287437  .0041069 -.0008004 
    5.2279426  -.1329443 -.0046128  .0054240  .0007122 -.0001295 
  292.4853419 11.0496805  .0373569  .0174504 -.0108844  .0042565 
     .0579959  -.4789099 -.0027643  .0024092  .0003160 -.0001260
   10.0344379  -.0366309 -.0103356 -.0003887  .0001631  .0000851 
   35.3103585  4.0202030  .0101963  .0008671  .0006986 -.0006386 
    -.4838089   .0420579  .0013107 -.0001041  .0001112 -.0000485 
   19.8187746  -.0448163 -.0019781  .0017314 -.0020253  .0007742 
  347.9901050  2.2198549  .0006474  .0027839 -.0026574  .0008337 
   -1.0393959  -.0550979  .0007576 -.0000038  .0000301 -.0000148 
   29.9328472  -.0057668 -.0003583  .0011605 -.0017321  .0007507 ]

NULLNAME QV_P2021
ARRY 12 6
[ 306.8551230 31.5865271  .7714122 -.0472867 -.0137630 -.0043244 
    -.5744886  -.6445307  .0717685  .0374925  .0006703 -.0011078 
    5.0960533  -.1236156  .0141271  .0071851 -.0009036  .0001141 
  303.6137622 11.1546585  .0657521  .0047931  .0046632 -.0023548 
    -.4223848  -.4765430  .0053044  .0027169  .0000218  .0000308 
    9.9871735  -.0574429 -.0104089  .0013747 -.0010359  .0004033 
   39.3527615  4.0428802  .0129105 -.0025246  .0030871 -.0011659 
    -.4403595   .0445777  .0012393 -.0000335 -.0000110  .0000109 
   19.7723296  -.0478387 -.0014164 -.0005968  .0008328 -.0004044 
  350.2176582  2.2230216  .0022203 -.0023881  .0028083 -.0011985 
   -1.0938714  -.0535429  .0008173 -.0000087  .0000111 -.0000001 
   29.9268841  -.0062024 -.0005211  .0002793 -.0002570  .0000426 ]

NULLNAME QV_P2022
ARRY 12 6
[ 339.1476882 32.9113174  .5060695 -.1187728 -.0354495  .0095128 
   -1.1101959  -.3913655  .1767019  .0296471 -.0064692 -.0004809 
    4.9929604  -.0768694  .0318479  .0036877 -.0002284 -.0002646 
  314.8412743 11.3076017  .0873280  .0099681 -.0049667  .0021224 
    -.8908538  -.4575420  .0137280  .0031794 -.0002048  .0000870
    9.9200638  -.0763029 -.0080128 -.0008734  .0014653 -.0005126 
   43.4079488  4.0677519  .0109318  .0041260 -.0039755  .0013748 
    -.3945761   .0469666  .0011124  .0000701 -.0000958  .0000318 
   19.7229060  -.0511244 -.0016867  .0003213 -.0006636  .0003205 
  352.4421218  2.2256340  .0002719  .0021329 -.0028021  .0011115 
   -1.1465947  -.0518896  .0008209  .0000933 -.0000896  .0000341 
   29.9202255  -.0072339 -.0001205 -.0008035  .0005720 -.0001362 ]

NULLNAME QV_P2023
ARRY 12 6
[  12.4203656 33.4726048  .0288015 -.1829167  .0099944 -.0010078 
   -1.3021626   .0227011  .2221304 -.0012170 -.0087196  .0004485 
    4.9511336  -.0043210  .0384200  .0015527 -.0026066  .0005278 
  326.2433278 11.5029149  .1044171  .0111777 -.0044895  .0010461 
   -1.3316062  -.4209363  .0229542  .0029128  .0002134 -.0000749 
    9.8358273  -.0916158 -.0070946  .0020000 -.0017546  .0007482 
   47.4881579  4.0928916  .0134868 -.0035210  .0048241 -.0021126 
    -.3464911   .0491749  .0011049 -.0001601  .0001422 -.0000562 
   19.6700732  -.0545761 -.0023104  .0011868 -.0010745  .0003239 
  354.6684701  2.2269149 -.0002141  .0003069  .0001917 -.0003173 
   -1.1976256  -.0501577  .0009063 -.0000366  .0000579 -.0000238 
   29.9125033  -.0082600 -.0010453  .0016981 -.0019454  .0007683 ]

NULLNAME QV_P2024
ARRY 12 6
[  45.7478419 33.0167364 -.4649779 -.1353636  .0149485  .0036406 
   -1.0668191   .4306968  .1709431 -.0302545 -.0055970  .0014732 
    4.9847066   .0693641  .0330463 -.0050572  .0000980 -.0002446 
  337.8583941 11.7324983  .1249764 -.0003094  .0052573 -.0021326 
   -1.7265370  -.3658077  .0322689  .0032320 -.0001439  .0000585 
    9.7381105  -.1031246 -.0045061  .0001035  .0008145 -.0003540
   51.5937268  4.1181401  .0122956  .0008849 -.0013419  .0006425 
    -.2962854   .0511951  .0009292  .0000420 -.0001118  .0000481 
   19.6136229  -.0583266 -.0013417 -.0011207  .0013155 -.0005128 
  356.8953523  2.2266074  .0003203 -.0017139  .0016708 -.0006067 
   -1.2468797  -.0483417  .0009265  .0000376 -.0000455  .0000218 
   29.9037191  -.0092326 -.0000327 -.0007969  .0010798 -.0004546 ]

NULLNAME QV_P2025
ARRY 12 6
[  78.2698292 31.7543549 -.7507255 -.0511148  .0303432 -.0048466 
    -.4977303   .6671174  .0605776 -.0387153  .0010356  .0005217 
    5.0822406   .1195829  .0156830 -.0049544 -.0016537  .0006311 
  349.7515396 11.9928729  .1320617  .0131542 -.0107448  .0037657 
   -2.0577285  -.2916285  .0415890  .0032586 -.0002831  .0000751 
    9.6307415  -.1103354 -.0022551  .0008725 -.0004414  .0002705 
   55.7356990  4.1433080  .0111440  .0018686 -.0009665  .0000679 
    -.2440376   .0529757  .0008056 -.0000329  .0000075 -.0000102 
   19.5534677  -.0616438 -.0020415  .0016863 -.0018582  .0007321 
  359.1277282  2.2258347 -.0018769  .0034605 -.0036112  .0013105 
   -1.2944081  -.0464433  .0009423  .0000657 -.0000560  .0000191 
   29.8942558  -.0096242  .0000726  .0003999 -.0006964  .0003620 ]

*" \end{Scommands}
******* END of QVSOP.s

