/* alifhdr.c */

/* add a LIF header to assembled files for transfer to HP-71B */
/* J-F Garnier, May 2006 */

#define _CRT_SECURE_NO_WARNINGS

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <string>

void addheader(char *inputfile, char *LIFfile, char ftype)
{
  FILE *fic;
  FILE *dest;
  long lg;
  int i, n;
  unsigned char s[32];
  char c;
  char buffer[256];

  fic=fopen(inputfile,"rb");
  if (fic==NULL) {
    printf("%s not found\n",inputfile);
    return;
  }

  /* find file size */
  fseek(fic,0,SEEK_END);
  lg=ftell(fic);
  fseek(fic,0,SEEK_SET);

  dest=fopen(LIFfile,"wb");
  if (dest==NULL) {
    printf("impossible to create %s\n",LIFfile);
    fclose(fic);
    return;
  }

  /* build LIF header */

  /* file name */
  for (i=0;i<10;i++) {
    c=LIFfile[i];
    if ((c==0)||(c=='.')) break;
    s[i]=toupper(c);
  }
  for (;i<10;i++) {
    s[i]=' ';
  }

  /* file type */
  switch (ftype) {
    case 'L':
      s[10]=0xE2;
      s[11]=0x08;
      break;
    case 'B':
      s[10]=0xE2;
      s[11]=0x04;
      break;
    case 'R':
      s[10]=0xE2;
      s[11]=0x1C;
      break;
    case 'T':
      s[10]=0x00;
      s[11]=0x01;
      break;
    default:
      s[10]=0;
      s[11]=0;
  }

  /* start sector */
  s[12]=0; s[13]=0; s[14]=0; s[15]=0;

  /* date&time */
  s[20]=0; s[21]=0; s[22]=0;
  s[23]=0; s[24]=0; s[25]=0;

  /* code */
  s[26]=0x80;
  s[27]=0x01;

  if (ftype == 'T')
  {
    char buffer[4096];
    std::string os;

    /* implementation field */
    s[28]=0;
    s[29]=0;
    s[30]=0;
    s[31]=0;

    /* read file content in lines */
    while (fgets(buffer,4096,fic) != NULL) {
      /* delete CR LF at the end */
      int nLength = strlen(buffer);
      while (nLength > 0 && (buffer[nLength-1] == 13 || buffer[nLength-1] == 10))
        --nLength;
      buffer[nLength] = 0;

      if (*buffer) {
        os += static_cast<unsigned char>(nLength >> 8);
        os += static_cast<unsigned char>(nLength & 0xff);
        os = os + buffer;

        /* add filler data */
        if ((nLength & 0x1) != 0) {
          os += static_cast<char>(0);
        }
      }
    }

    lg = os.length(); /* actual data length */

    /* fill until end of block boundary */
    while ((lg & 0xff) != 0) {
      os += static_cast<char>(-1);
      ++lg;
    }

    /* len in sectors */
    s[16]=0;
    s[17]= static_cast<unsigned char>(lg >> 24);
    s[18]= static_cast<unsigned char>(lg >> 16);
    s[19]= static_cast<unsigned char>(lg >> 8);

    /* write header */
    fwrite(s,1,32,dest);

    /* and append file content */
    fwrite(os.c_str(),1,lg,dest); /* write all */
  }
  else /* L, B, R */
  {
    /* length in nibbles */
    s[28]= static_cast<unsigned char>(lg << 1);
    s[29]= static_cast<unsigned char>(lg >> 7);
    s[30]= static_cast<unsigned char>(lg >> 15);
    s[31]=0;

    /* len in sectors */
    lg = lg+255;
    s[16]=0;
    s[17]= static_cast<unsigned char>(lg >> 24);
    s[18]= static_cast<unsigned char>(lg >> 16);
    s[19]= static_cast<unsigned char>(lg >> 8);

    /* write header */
    fwrite(s,1,32,dest);

    /* and append file content */
    do {
      memset(buffer,0xff,256);
      n=fread(buffer,1,256,fic);
      if (n<=0) break;
      fwrite(buffer,1,256,dest); /* always write entire sector */
    } while (1);
  }

  fclose(dest);
  fclose(fic);
}

void removeheader(char *inputfile, char *LIFfile)
{
  FILE *fic;
  FILE *dest;
  long len;
  int n;
  unsigned char s[32];
  char buffer[256];

  fic=fopen(inputfile,"rb");
  if (fic==NULL) {
    printf("%s not found\n",inputfile);
    return;
  }

  dest=fopen(LIFfile,"wb");
  if (dest==NULL) {
    printf("impossible to create %s\n",LIFfile);
    fclose(fic);
    return;
  }

  /* read header */
  fread(s,1,32,fic);
  /* length in nibbles */
  len = s[30];
  len <<= 8;
  len += s[29];
  len <<= 8;
  len += s[28];
  len = (len+1)/2; /* rounded to bytes */

  /* and copy file content */
  do {
    n=256;
    if (n>len) n=len;
    len -= n;
    fread(buffer,1,n,fic);
    fwrite(buffer,1,n,dest);
  } while (len>0);

  fclose(dest);
  fclose(fic);
}

void viewheader(char *name)
{
  FILE *fic;
  int n;
  long len;
  unsigned char buffer[32];
  char s[16];
  char f;

  fic=fopen(name,"rb");
  if (fic==NULL) {
    printf("%s not found\n",name);
    return;
  }

  f=0;
  n=fread(buffer,1,32,fic);
  if (n!=32) {
    printf("read error in %s\n",name);
    fclose(fic);
    return;
  }

  strncpy(s,reinterpret_cast<char *>(buffer),10);
  s[10]=0;
  printf("File name = %s\n",s);

  n  = buffer[10]<<8;
  n += buffer[11];
  printf("File type = %04X\n",n);

  if ((buffer[12]!=0)||(buffer[13]!=0)) f=1;
  n  = buffer[14]<<8;
  n += buffer[15];
  printf("File start sector   = %d\n",n);
  if (n!=0) f=1;

  if ((buffer[16]!=0)||(buffer[17]!=0)) f=1;
  n  = buffer[18]<<8;
  n += buffer[19];
  printf("File len in sectors = %d\n",n);

  n  = buffer[26]<<8;
  n += buffer[27];
  printf("Implementation code = %04X\n",n);
  if (n!=0x8001) f=1;

  len = buffer[30];
  len <<= 8;
  len += buffer[29];
  len <<= 8;
  len += buffer[28];
  printf("File length = %ld nibbles (%ld bytes)\n",len,(len+1)/2);
  if (buffer[31]!=0) f=1;

  if (f)
    printf("Warning: probably not a valid LIF header\n");
}

void help(void)
{
  printf("ALIFHDR: add a 32-byte LIF header for file transfer to HP-71B\n");
  printf("Syntax : ALIFHDR input_file LIF_file /[L|B|R|T]\n");
  printf("         ALIFHDR LIF_file /i\n");
  printf("         ALIFHDR LIF_file output_file /x\n");
  printf("Options :\n");
  printf("  /L : use HP-71B LEX  file type (default)\n");
  printf("  /B : use HP-71B BIN  file type\n");
  printf("  /R : use HP-71B ROM  file type\n");
  printf("  /T : use HP-71B TEXT file type with data conversion\n");
  printf("  /i : display header information of LIF file\n");
  printf("  /x : remove header from LIF file\n");
  printf("  /h : help\n");
  printf("V1.2 (c) J-F. Garnier & C. Giesselink, 2014.\n");
}

void main(int argc,char *argv[])
{
  char argument[32];
  char filename[32];
  char LIFname[32];
  char ftype, finfo, fhelp, frem;
  int i;

  fhelp=0; finfo=0; ftype=0; frem=0;

  LIFname[0]=0;
  filename[0]=0;
  ftype='L';

  for (i=1;i<argc;i++) {
    strcpy(argument,argv[i]);
    if (argument[0]=='/') {
      argument[1]=toupper(argument[1]);
      switch (argument[1]) {
  case 'H':
  case '?': fhelp=1; break;
  case 'B': ftype='B'; break;
  case 'R': ftype='R'; break;
  case 'T': ftype='T'; break;
  case 'I': finfo=1; break;
  case 'X': frem=1; break;
      }
    }
    else {
      if (filename[0]==0)
        strcpy(filename,argument);
      else
        strcpy(LIFname,argument);
    }
  }

  if (fhelp) {
    help();
    exit(0);
  }

  if (filename[0]==0) {
    printf("Error: no file name\n");
    exit(0);
  }
  if ((!finfo)&&(!frem)&&(LIFname[0]==0)) {
    printf("Error: no destination file name\n");
    exit(0);
  }

  if (finfo)
    viewheader(filename);
  else if (frem)
    removeheader(filename, LIFname);
  else
    addheader(filename, LIFname, ftype);
}
