# tinyturtle v0.91
# by Piotr Kowalewski (komame)
# December 2025

import hpprime as hp
import math

#-------------------- turtle --------------------
class Turtle:
    _FIXED_POINT_SHIFT = 16
    _FIXED_POINT_SCALE = 1 << _FIXED_POINT_SHIFT
    _FIXED_POINT_CEIL_ADD = _FIXED_POINT_SCALE - 1

    _SPEED_MAP = {
        'fastest': 0, 'fast': 2, 'normal': 5, 'slow': 10, 'slowest': 20,
        0: 0, 10: 1, 9: 2, 8: 3, 7: 4, 6: 5, 5: 7, 4: 9, 3: 12, 2: 15, 1: 20
    }
    
    _COLOR_NAMES = {
        'black':(0,0,0), 'white':(255,255,255), 'red':(255,0,0), 'lime':(0,255,0),
        'blue':(0,0,255), 'yellow':(255,255,0), 'cyan':(0,255,255), 'magenta':(255,0,255),
        'silver':(192,192,192), 'gray':(128,128,128), 'grey':(128,128,128),
        'maroon':(128,0,0), 'olive':(128,128,0), 'green':(0,128,0), 
        'purple':(128,0,128), 'teal':(0,128,128), 'navy':(0,0,128),
        'orange':(255,165,0), 'brown':(165,42,42), 'pink':(255,192,203),
        'gold':(255,215,0), 'violet':(238,130,238), 'indigo':(75,0,130),
        'turquoise':(64,224,208), 'coral':(255,127,80), 'salmon':(250,128,114),
        'beige':(245,245,220), 'khaki':(240,230,140), 'lavender':(230,230,250),
        'crimson':(220,20,60), 'plum':(221,160,221), 'tomato':(255,99,71),
        'darkgreen':(0,100,0), 'darkblue':(0,0,139), 'darkred':(139,0,0),
        'lightblue':(173,216,230), 'lightgreen':(144,238,144), 'lightgray':(211,211,211)
    }

    _FONT_DATA = (0x00000000,0x44444040,0xAA000000,0x0AFAFA00,0x278611E4,0x0CD24B30,0x4AA4BA50,0x44000000,0x24444420,0x21111120,0x00A4A000,0x004E4000,0x00000048,0x000F0000,0x00000040,0x11224480,0x69BD9960,0x26222220,0x691124F0,0x69161960,0x26AAF220,0xF88E11E0,0x698E9960,0xF1122440,0x69969960,0x69971960,0x00400040,0x00400048,0x02484200,0x00E0E000,0x08424800,0x4A224040,0x069BB870,0x6999F990,0xE99E99E0,0x69988960,0xE99999E0,0xF88E88F0,0xF88E8880,0x788B9960,0x999F9990,0xE44444E0,0x11111960,0x99ACA990,0x888888F0,0x9FF99990,0x9DDBB990,0x69999960,0xE999E880,0x69999A50,0xE999E990,0x69861960,0xE4444440,0x99999960,0x99996660,0x9999FF90,0x99969990,0x99962440,0xF12488F0,0x74444470,0x88442210,0xE22222E0,0x04A10000,0x0000000F,0x84000000,0x00617970,0x88E999E0,0x00788870,0x11799970,0x0069F870,0x344E4440,0x0069971E,0x88E99990,0x40444440,0x40444448,0x889ACA90,0xC44444E0,0x009FF990,0x00E99990,0x00699960,0x00E99E88,0x00799711,0x00E98880,0x007861E0,0x44E44420,0x00999970,0x00999520,0x0099FF90,0x00996990,0x0099971E,0x00F168F0,0x12266221,0x44444444,0x84466448,0x005B0000)

    def __init__(self, screen_g=0, buffer_g=1, visible=True, viewport=False, grid=0):
        self.last_keyboard_state = hp.keyboard()
        self.screen_g, self.buffer_g = screen_g, buffer_g
        
        raw_mode = 0
        self.grid_color = None

        if isinstance(grid, (tuple, list)):
            if len(grid) >= 1: raw_mode = int(grid[0])
            self._temp_grid_color_arg = grid[1] if len(grid) >= 2 else None
        else:
            raw_mode = int(grid)
            self._temp_grid_color_arg = None
            
        self.grid_mode = raw_mode

        self.viewport = bool(viewport)
        self._display_list = []
        self.delay_ms = 0
        
        self._tracer_step = 1
        self._tracer_counter = 0
        
        self.is_visible = visible 

        self.W, self.H = 320, 240 # hp.grobw(self.screen_g), hp.grobh(self.screen_g)
        self._last_frame_ticks = hp.ticks()       

        self.x = 0.0
        self.y = 0.0
        
        self._heading = 0.0 
        
        self.pen, self.pen_width, self.pen_color = True, 1, 0x000000
        self.fill_color, self.fill_active = None, False
        self.fill_points, self._dirty, self._outline_ops = [], None, None
        self._colormode = 255.0
        self._background_color = 0xFFFFFF

        if self._temp_grid_color_arg is not None:
            self.grid_color = self._parse_color(self._temp_grid_color_arg)

        if not self.viewport:
            self._ensure_buffer(self._background_color)
            if self.screen_g != 0:
                hp.dimgrob(self.screen_g, self.W, self.H, self._background_color)
            if self.grid_mode > 0:
                self._render_grid(self.buffer_g, 0.0, 0.0, 1.0)
            self._present_full()

    def tracer(self, n=None, delay=None):
        if n is None: return self._tracer_step
        self._tracer_step = int(n)
        self._tracer_counter = 0
        if delay is not None: self.delay_ms = int(delay)

    def update(self):
        if not self.viewport:
            self._present_full()

    def hideturtle(self):
        if not self.is_visible: return
        self._mark_turtle_dirty()
        self.is_visible = False
        self._present_dirty()
    ht = hideturtle

    def showturtle(self):
        if self.is_visible: return
        self.is_visible = True
        self._mark_turtle_dirty()
        self._present_dirty()
    st = showturtle

    def isvisible(self): return self.is_visible

    def heading(self): 
        return float(self._heading)
    getheading = heading

    # ---------- Public API ----------
    def home(self):
        self.x = 0.0
        self.y = 0.0
        self.setheading(0.0)

    def reset(self):
        if self.viewport:
            self._display_list = [] 
        else:
            hp.dimgrob(self.buffer_g, self.W, self.H, self._background_color)
            
            if self.grid_mode > 0:
                self._render_grid(self.buffer_g, 0.0, 0.0, 1.0)
            
            if self.buffer_g != self.screen_g:
                hp.blit(self.screen_g, 0, 0, self.buffer_g)
        
        self.home()
        self.pen, self.pen_width, self.pen_color = True, 1, 0
        self.fill_color = self.pen_color
        self.fill_active, self.fill_points, self._dirty, self._outline_ops = False, [], None, None

    def speed(self, s=None):
        if self.viewport: return 0
        if s is None:
            for k, v in self._SPEED_MAP.items():
                if v == self.delay_ms and isinstance(k, str): return k
            return self.delay_ms
        if s in self._SPEED_MAP: self.delay_ms = self._SPEED_MAP[s]
        else: raise ValueError("speed() values: 0-10 or strings")

    def penup(self): self.pen = False
    pu = up = penup

    def pendown(self): self.pen = True
    pd = down = pendown

    def width(self, w=None):
        if w is None: 
            return int(self.pen_width)
        self.pen_width = max(1, int(round(w)))
    pensize = width

    def colormode(self, m):
        if m not in (1.0, 255, 255.0): raise ValueError("colormode: 1.0 or 255 supported")
        self._colormode = float(m)
        return self._colormode

    def bgcolor(self, *args):
        argc = len(args)
        if argc == 0: 
            return self._int2rgb(self._background_color)
        elif argc == 1: new_c = self._parse_color(args[0])
        elif argc == 3: new_c = self._parse_color(args)
        else: raise TypeError("bgcolor takes 0, 1, or 3 args")
        
        self._background_color = int(new_c)
        if not self.viewport: self.clear()

    def color(self, *args):
        argc = len(args)
        if argc == 0: 
            return (self._int2rgb(self.pen_color), self._int2rgb(self.fill_color))
        elif argc == 1: c=int(self._parse_color(args[0])); self.pen_color=c; self.fill_color=c
        elif argc == 2: self.pen_color=int(self._parse_color(args[0])); self.fill_color=int(self._parse_color(args[1]))
        elif argc == 3: c=int(self._parse_color(args)); self.pen_color=c; self.fill_color=c
        elif argc == 6: self.pen_color=int(self._parse_color(args[0:3])); self.fill_color=int(self._parse_color(args[3:6]))
        else: raise TypeError("color takes 0, 1, 2, 3, or 6 args")

    def pencolor(self, *args):
        a = len(args)
        if a == 0: 
            return self._int2rgb(self.pen_color)
        elif a == 1: c = self._parse_color(args[0])
        elif a == 3: c = self._parse_color(args)
        else: raise TypeError("pencolor takes 0, 1 or 3 args")
        self.pen_color = int(c)

    def fillcolor(self, *args):
        a = len(args)
        if a == 0: 
            return self._int2rgb(self.fill_color)
        elif a == 1: c = self._parse_color(args[0])
        elif a == 3: c = self._parse_color(args)
        else: raise TypeError("fillcolor takes 0, 1 or 3 args")
        self.fill_color = int(c)

    def setheading(self, deg): 
        if self.is_visible: self._mark_turtle_dirty()
        self._heading = float(deg) % 360.0
        if self.is_visible: self._mark_turtle_dirty()
        self._present_dirty()
    seth = setheading

    def right(self, deg): self.setheading(self.heading() - float(deg))
    rt = right

    def left(self, deg): self.setheading(self.heading() + float(deg))
    lt = left

    def goto(self, x, y=None):
        if y is None:
            try:
                x, y = x[0], x[1]
            except:
                raise TypeError("goto requires x,y or (x,y)")
        x_virt, y_virt = float(x), float(y)
        start_pt_virt = (self.x, self.y)

        if getattr(self, 'is_visible', True): 
            self._mark_turtle_dirty()

        if self.fill_active:
            end_pt_phys = self._v2p_pt([x_virt, y_virt])
            if not self.fill_points or self.fill_points[-1] != end_pt_phys:
                self.fill_points.append(end_pt_phys)
            
            if self.pen:
                start_pt_phys = self._v2p_pt(start_pt_virt)
                op_data = ('polyline', [start_pt_phys, end_pt_phys], self.pen_width, self.pen_color)
                self._outline_ops.append(op_data)

        self.x, self.y = x_virt, y_virt

        if self.pen:
            should_render = not (self.viewport and self.fill_active)

            if should_render:
                start_pt_phys = self._v2p_pt(start_pt_virt)
                end_pt_phys = self._v2p_pt([x_virt, y_virt])
                bbox = self._render_thick_polyline([start_pt_phys, end_pt_phys], self.pen_width, self.pen_color)
                if bbox and not self.viewport: 
                    self._mark_dirty(*bbox)
        
        if getattr(self, 'is_visible', True):
            self._mark_turtle_dirty()

        if not self.viewport:
            self._present_dirty()
    setpos = setposition = goto

    def forward(self, dist):
        if self.viewport or self.delay_ms == 0:
            rad = self.heading() * (math.pi / 180.0)
            self.goto(self.x + math.cos(rad) * dist, self.y + math.sin(rad) * dist)
            return

        start_x_virt, start_y_virt = self.x, self.y
        step_len = 2
        target_dist = float(dist)
        distance_covered = 0.0
        rad = self.heading() * (math.pi / 180.0)
        dx = math.cos(rad)
        dy = math.sin(rad)
        
        direction = 1.0 if target_dist >= 0 else -1.0
        total_abs = abs(target_dist)
        
        while distance_covered < total_abs:
            remaining = total_abs - distance_covered
            step = min(step_len, remaining)
            self.goto(self.x + dx * step * direction, self.y + dy * step * direction)
            distance_covered += step
            
        if self.pen and not self.viewport:
            p_start = self._v2p_pt([start_x_virt, start_y_virt])
            p_end = self._v2p_pt([self.x, self.y])
            
            bbox = self._render_thick_polyline([p_start, p_end], self.pen_width, self.pen_color)
            
            if bbox:
                self._mark_dirty(*bbox)
                self._present_dirty()
    fd = forward

    def backward(self, dist): return self.forward(-dist)
    bk = back = backward

    def circle(self, radius, extent=None, steps=None):
        if extent is None: extent = 360.0
        R = abs(float(radius))
        extent_mag = abs(float(extent))
        if R == 0.0 or extent_mag == 0.0: return
        turn_dir = 1.0 if radius >= 0 else -1.0
        if steps is not None:
            count = int(steps)
            if count < 1: count = 1
        else:
            _seg_len_px = 5.0 
            arc_len = (2 * math.pi * R) * (extent_mag / 360.0)
            count = int(arc_len / _seg_len_px)
            count = max(4, count)
            
        steps = count
        PI, h0, x0_v, y0_v = math.pi, self.heading(), self.x, self.y
        th0_r = h0 * (PI/180.0)
        
        cx_v, cy_v = x0_v - math.sin(th0_r) * radius, y0_v + math.cos(th0_r) * radius
        vx_v, vy_v = x0_v - cx_v, y0_v - cy_v
        start_angle_r = math.atan2(vy_v, vx_v)
        total_angle_r = (extent_mag * (PI/180.0)) * turn_dir
        
        pts_v = []
        for i in range(steps + 1):
            t = i / float(steps)
            theta = start_angle_r + total_angle_r * t
            
            px = cx_v + R * math.cos(theta)
            py = cy_v + R * math.sin(theta)
            pts_v.append([px, py])

        pts_phys = self._v2p_points(pts_v)
        center_phys = self._v2p_pt([cx_v, cy_v])

        if self.fill_active:
            self.fill_points.extend(pts_phys[1:])
            if self.pen:
                if steps < 12: 
                    op_data = ('polyline', pts_phys, self.pen_width, self.pen_color)
                else:
                    op_data = ('arc', pts_phys, center_phys, self.pen_width, self.pen_color)
                self._outline_ops.append(op_data)

        if self.viewport or self.delay_ms == 0:
            self.x, self.y = pts_v[-1]
            self.setheading((h0 + (extent * turn_dir)) % 360.0)
            
            if self.pen:
                should_render = not (self.viewport and self.fill_active)
                if should_render:
                    is_closed = (abs(extent) % 360.0 == 0)
                    
                    if steps < 12:
                        bbox = self._render_thick_polyline(pts_phys, self.pen_width, self.pen_color, closed=is_closed)
                    else:
                        bbox = self._render_thick_arc(pts_phys, center_phys, self.pen_width, self.pen_color, is_closed=is_closed)
                        
                    if bbox and not self.viewport:
                        self._mark_dirty(*bbox); self._present_dirty()
            return

        was_filling = self.fill_active
        self.fill_active = False 
        
        try:
            angle_step = (extent * turn_dir) / steps
            
            for i in range(1, len(pts_v)):
                target_pt = pts_v[i]
                self.goto(target_pt[0], target_pt[1])
                self._heading = (self._heading + angle_step) % 360.0
        finally:
            self.fill_active = was_filling
            
        self.setheading((h0 + (extent * turn_dir)) % 360.0)

    def begin_fill(self):
        self.fill_active = True
        self.fill_points = [self._v2p_pt([self.x, self.y])]
        self._outline_ops = []

    def end_fill(self, fill_color=None):
        if not self.fill_active: return
        
        saved_delay = self.delay_ms
        self.delay_ms = 0
        
        try:
            self.fill_active = False
            outline_ops, fill_pts = self._outline_ops, self.fill_points
            self.fill_points, self._outline_ops = [], None
            
            union_bb = None
            def update_union_bb(bb):
                nonlocal union_bb
                if not bb: return
                if union_bb is None: union_bb = list(bb)
                else:
                    union_bb[0]=min(union_bb[0],bb[0]); union_bb[1]=min(union_bb[1],bb[1])
                    union_bb[2]=max(union_bb[2],bb[2]); union_bb[3]=max(union_bb[3],bb[3])
            
            if len(fill_pts) >= 3:
                poly = fill_pts[:]
                if poly[0] != poly[-1]: poly.append(poly[0])
                fc = self.fill_color if fill_color is None else self._parse_color(fill_color)
                
                self._fill_polygon_aet_fxp(poly, fc, g=self.buffer_g)
                if not self.viewport: update_union_bb(self._bbox_of_points(poly))
            
            if outline_ops:
                for op in outline_ops:
                    op_type = op[0]
                    bb = None
                    if op_type == 'arc':
                        _, points, center, width, color = op
                        if len(points) < 2: continue
                        is_closed = (abs(self._v2p_pt(self.pos())[0] - center[0]) < 1 and
                                    abs(self._v2p_pt(self.pos())[1] - center[1]) < 1)
                        bb = self._render_thick_arc(points, center, width, color, is_closed=is_closed)

                    elif op_type == 'polyline':
                        _, points, width, color = op
                        if len(points) < 2: continue
                        bb = self._render_thick_polyline(points, width, color, closed=False)
                    
                    if not self.viewport: update_union_bb(bb)
                    
            if union_bb and not self.viewport: 
                self._mark_dirty(*union_bb)
                self._present_dirty()
                
        finally:
            self.delay_ms = saved_delay

    def cancel_fill(self): self.fill_active, self.fill_points, self._outline_ops = False, [], None
    
    def dot(self, size=None, color=None):
        r = (size if size is not None else self.pen_width) / 2.0
        col = self.pen_color if color is None else self._parse_color(color)
        
        cx, cy = self._v2p_pt([self.x, self.y])
        
        if self.viewport:
             norm_pos = self._p2n_pt([cx, cy])
             self._display_list.append(('dot', norm_pos, r, col))
             return

        self._fill_disk_poly_g(self.buffer_g, cx, cy, r, col)
        
        bbox = self._bbox_disk(cx, cy, r)
        self._mark_dirty(*bbox)
        self._present_dirty()

    stamp_dot = dot

    def clear(self):
        if self.viewport:
            self._display_list = []
        else:
            hp.dimgrob(self.buffer_g, self.W, self.H, self._background_color)
            if self.grid_mode > 0:
                self._render_grid(self.buffer_g, 0.0, 0.0, 1.0)
            self._present_full()

    def write(self, arg, move=False, align="left", font_size=1):
            text = str(arg)
            # Szerokość znaku w definicji fontu to 5, wysokość to 8
            total_width_units = len(text) * 5 
            total_height_units = 8
            
            # Oblicz całkowitą szerokość tekstu w jednostkach wirtualnych
            text_pixel_width = total_width_units * font_size
            
            # Obliczanie pozycji startowej X
            # Kluczowe: użycie int() eliminuje ułamki (np. .5), które psują rendering fontu
            if align == "center":
                start_x = int(self.x - (text_pixel_width / 2))
            elif align == "right":
                start_x = int(self.x - text_pixel_width)
            else:
                start_x = int(self.x)
            
            # Obliczanie pozycji startowej Y (int dla bezpieczeństwa)
            # Oryginalna logika przesuwa tekst w górę o jego wysokość
            start_y = int(self.y + (total_height_units * font_size))
            
            color = self.pen_color

            if self.viewport:
                # W trybie viewport zapisujemy do listy "czyste" współrzędne całkowite
                norm_pos = [start_x, start_y]
                
                self._display_list.append(('bmp_text', text, norm_pos, font_size, color))
                
                if move: 
                    self._update_turtle_position_after_write(text_pixel_width, align)
                return

            # Tryb bezpośredniego rysowania (bez viewport)
            phys_x, phys_y = self._v2p_pt([start_x, start_y])
            
            self._draw_bitmap_text(self.buffer_g, text, phys_x, phys_y, font_size, color, scale=1.0)
            
            if move:
                self._update_turtle_position_after_write(text_pixel_width, align)
                
            self._present_full()

    def position(self): return (self.x, self.y)
    pos = position

    def towards(self, x, y=None):
        if y is None:
            if isinstance(x, (tuple, list)) and len(x) >= 2:
                x, y = x[0], x[1]
            else:
                raise TypeError("towards() requires two numbers or a tuple (x,y)")        
        return math.degrees(math.atan2(y - self.y, x - self.x))

    def distance(self, x, y=None):
        if y is None:
            if isinstance(x, (tuple, list)) and len(x) >= 2:
                x, y = x[0], x[1]
            else:
                raise TypeError("distance() requires two numbers or a tuple (x,y)")
        return math.sqrt((x - self.x)**2 + (y - self.y)**2)

    def xcor(self): return self.x
    def ycor(self): return self.y    

    def wait(self, ms):
        if self.viewport: return 
        if ms>0:
            end=hp.ticks()+ms
            while hp.ticks()<end:pass

    def show(self):
        if not self.viewport: return
        offset_x, offset_y = 0.0, 0.0
        scale = 1.0
        zoom_level = 0
        
        if self.buffer_g != self.screen_g:
            hp.dimgrob(self.buffer_g, self.W, self.H, self._background_color)

        self._render_viewport_frame(offset_x, offset_y, scale)

        running = True
        while running:
            key = self.read_key() 
            if key:
                dirty = False
                if key == 8:    # left
                    offset_x -= 20 / scale; dirty = True
                elif key == 7:  # right
                    offset_x += 20 / scale; dirty = True
                elif key == 12:  # up
                    offset_y += 20 / scale; dirty = True
                elif key == 2: # down
                    offset_y -= 20 / scale; dirty = True
                elif key == 50: # plus (zoom In)
                    if zoom_level < 5:
                        scale *= 1.2; zoom_level += 1; dirty = True
                elif key == 45: # minus (zoom Out)
                    if zoom_level > -5:
                        scale /= 1.2; zoom_level -= 1; dirty = True
                elif key in (30, 4, 14): # Enter, Esc, Backspace
                    running = False

                if dirty:
                    self._render_viewport_frame(offset_x, offset_y, scale)
            t_wait = hp.ticks() + 10
            while hp.ticks() < t_wait: pass

    def read_key(self):
        while True:
            current_state = hp.keyboard()
            changed_keys = current_state ^ self.last_keyboard_state
            if changed_keys:
                self.last_keyboard_state = current_state
                for i in range(52):
                    if changed_keys & (1 << i):
                        if current_state & (1 << i): return i

    def done(self):
        try:
            self.read_key()
        except:
            pass
    # -------------- Internal methods --------------

    def _get_turtle_poly_phys(self):
        SIZE = 10 
        
        rad = math.radians(self.heading())
        cos_a, sin_a = math.cos(rad), math.sin(rad)
        cx, cy = self._v2p_pt([self.x, self.y])
        
        pts = [
            (SIZE, 0),
            (-SIZE*0.5, SIZE*0.5),
            (-SIZE*0.5, -SIZE*0.5)
        ]
        
        poly_phys = []
        for px, py in pts:
            rot_x = px * cos_a - py * sin_a
            rot_y = px * sin_a + py * cos_a
            poly_phys.append([cx + rot_x, cy - rot_y])
            
        return poly_phys

    def _mark_turtle_dirty(self):
        if self.viewport: return
        poly = self._get_turtle_poly_phys()
        bbox = self._bbox_of_points(poly)
        self._mark_dirty(bbox[0]-2, bbox[1]-2, bbox[2]+2, bbox[3]+2)

    def _v2p_y(self, y_virt): return self.H / 2.0 - y_virt
    def _v2p_pt(self, pt_virt): return [pt_virt[0] + self.W / 2.0, self.H / 2.0 - pt_virt[1]]
    def _v2p_points(self, points_virt):
        w_half, h_half = self.W / 2.0, self.H / 2.0
        return [[p[0] + w_half, h_half - p[1]] for p in points_virt]

    def _p2n_pt(self, pt_phys): return [pt_phys[0] - self.W / 2.0, self.H / 2.0 - pt_phys[1]]
    def _p2n_points(self, points_phys):
        w_half, h_half = self.W / 2.0, self.H / 2.0
        return [[p[0] - w_half, h_half - p[1]] for p in points_phys]

    def _render_grid(self, g, off_x, off_y, scale):
        if self.grid_mode == 0: return
        W, H = self.W, self.H
        cx, cy = W / 2.0, H / 2.0

        if self.grid_mode == 1: step_x, step_y = 10.0, 10.0
        elif self.grid_mode == 2: step_x, step_y = 20.0, 20.0
        else: step_x, step_y = 20.0, 20.0 * 0.866025 

        min_virt_x = (0 - cx) / scale - off_x
        max_virt_x = (W - cx) / scale - off_x
        v_h = H / scale
        center_virt_y = -off_y 
        min_virt_y = center_virt_y - v_h/1.0
        max_virt_y = center_virt_y + v_h/1.0

        start_row = int(math.floor(min_virt_y / step_y))
        end_row = int(math.ceil(max_virt_y / step_y))
        start_col = int(math.floor(min_virt_x / step_x))
        end_col = int(math.ceil(max_virt_x / step_x))

        default_color = 0xA0A0A0
        
        color = self.grid_color if self.grid_color is not None else default_color

        if self.grid_mode == 2:
            for col in range(start_col, end_col + 1):
                vx = col * step_x
                sx = int((vx + off_x) * scale + cx)
                if 0 <= sx < W: hp.line(g, sx, 0, sx, H - 1, color)
            for row in range(start_row, end_row + 1):
                vy = row * step_y
                sy = int(cy - (vy + off_y) * scale)
                if 0 <= sy < H: hp.line(g, 0, sy, W - 1, sy, color)
        else:
            for row in range(start_row, end_row + 1):
                vy = row * step_y
                sy = int(cy - (vy + off_y) * scale)
                if not (0 <= sy < H): continue
                current_x_shift = 0.0
                if self.grid_mode == 3:
                    if row % 2 != 0: current_x_shift = step_x / 2.0
                for col in range(start_col - 1, end_col + 1):
                    vx = col * step_x + current_x_shift
                    sx = int((vx + off_x) * scale + cx)
                    if 0 <= sx < W: hp.pixon(g, sx, sy, color)

    def _draw_bitmap_text(self, g, text, x_origin, y_origin, font_size, color, scale=1.0):
        real_scale = float(font_size) * float(scale)
        
        if real_scale < 0.2: 
            return

        c_int = int(color)
        font_data = self._FONT_DATA
        
        cur_char_x = float(x_origin)
        start_y = float(y_origin)
        
        char_advance = 5.0 * real_scale

        for char in text:
            code = ord(char)
            if code < 32 or code > 126: code = 32
            
            idx = code - 32
            if idx >= len(font_data):
                char_data = 0
            else:
                char_data = font_data[idx]
            
            if char_data == 0:
                cur_char_x += char_advance
                continue

            for row in range(8):
                shift = 28 - (row * 4)
                row_bits = (char_data >> shift) & 0x0F
                
                if row_bits == 0: continue

                y1 = start_y + (row * real_scale)
                y2 = start_y + ((row + 1) * real_scale)
                
                scr_y = int(round(y1))
                scr_h = int(round(y2)) - scr_y
                
                if scr_h <= 0: continue

                for col in range(4):
                    if (row_bits >> (3 - col)) & 1:
                        x1 = cur_char_x + (col * real_scale)
                        x2 = cur_char_x + ((col + 1) * real_scale)
                        
                        scr_x = int(round(x1))
                        scr_w = int(round(x2)) - scr_x
                        
                        if scr_w > 0:
                            hp.fillrect(g, scr_x, scr_y, scr_w, scr_h, c_int, c_int)
            
            cur_char_x += char_advance
            
    def _update_turtle_position_after_write(self, text_width_px, align):
        if align == "left": self.x += text_width_px
        elif align == "center": self.x += text_width_px / 2.0
        elif align == "right": pass

    def _render_viewport_frame(self, offset_x, offset_y, scale):
        hp.dimgrob(self.buffer_g, self.W, self.H, self._background_color)
        if self.grid_mode > 0:
            self._render_grid(self.buffer_g, offset_x, offset_y, scale)
            
        self.viewport = False
        cx, cy = self.W / 2.0, self.H / 2.0
        try:        
            for op in self._display_list:
                op_type = op[0]
                
                if op_type == 'polyline':
                    _, norm_pts, w, c, closed = op
                    scr_pts = [[(p[0] + offset_x) * scale + cx, cy - (p[1] + offset_y) * scale] for p in norm_pts]
                    self._render_thick_polyline(scr_pts, max(1, w * scale), c, closed)

                elif op_type == 'polygon':
                    _, norm_pts, c = op
                    scr_pts = [[(p[0] + offset_x) * scale + cx, cy - (p[1] + offset_y) * scale] for p in norm_pts]
                    self._fill_polygon_aet_fxp(scr_pts, c, g=self.buffer_g)

                elif op_type == 'arc':
                    _, norm_pts, norm_center, w, c, is_closed = op
                    scr_pts = [[(p[0] + offset_x) * scale + cx, cy - (p[1] + offset_y) * scale] for p in norm_pts]
                    scr_center = [(norm_center[0] + offset_x) * scale + cx, cy - (norm_center[1] + offset_y) * scale]
                    
                    self._render_thick_arc(scr_pts, scr_center, max(1, w * scale), c, is_closed)

                elif op_type == 'dot':
                    _, norm_pos, base_r, col = op
                    
                    scr_x = (norm_pos[0] + offset_x) * scale + cx
                    scr_y = cy - (norm_pos[1] + offset_y) * scale                    
                    scr_r = base_r * scale
                    
                    self._fill_disk_poly_g(self.buffer_g, scr_x, scr_y, scr_r, col)

                elif op_type == 'bmp_text':
                    _, txt, norm_pos, f_size, col = op
                    
                    scr_x = (norm_pos[0] + offset_x) * scale + cx
                    scr_y = cy - (norm_pos[1] + offset_y) * scale
                    
                    self._draw_bitmap_text(self.buffer_g, txt, scr_x, scr_y, f_size, col, scale=scale)
        finally:

            self.viewport = True
            
        hp.blit(self.screen_g, 0, 0, self.buffer_g)

    def _render_thick_arc(self, points, center_phys, thickness, color, is_closed=False):
        if self.viewport:
            norm_pts = self._p2n_points(points)
            norm_center = self._p2n_pt(center_phys)
            self._display_list.append(('arc', norm_pts, norm_center, thickness, color, is_closed))
            return (0,0,0,0)

        t = int(round(thickness))
        if t <= 1:
            num_segs = len(points) - 1
            for i in range(num_segs):
                p1, p2 = points[i], points[i+1]
                hp.line(self.buffer_g, int(p1[0]+0.5), int(p1[1]+0.5), int(p2[0]+0.5), int(p2[1]+0.5), color)
            return self._bbox_of_points(points)

        radius = t / 2.0
        outer_pts, inner_pts = [], []
        cx, cy = center_phys

        for px, py in points:
            vx, vy = px - cx, py - cy
            mag = (vx*vx + vy*vy)**0.5
            if mag == 0: continue 
            nx, ny = vx / mag, vy / mag
            outer_pts.append([px + nx * radius, py + ny * radius])
            inner_pts.append([px - nx * radius, py - ny * radius])
        
        if not outer_pts: return (0,0,0,0)
        full_poly = outer_pts + inner_pts[::-1]
        self._fill_polygon_aet_fxp(full_poly, color, g=self.buffer_g)
        union_bb = self._bbox_of_points(full_poly)

        if not is_closed:
            start_pt = points[0]
            self._fill_disk_poly_g(self.buffer_g, start_pt[0], start_pt[1], radius, color)
            bb_start = self._bbox_disk(start_pt[0], start_pt[1], radius)
            union_bb = (min(union_bb[0], bb_start[0]), min(union_bb[1], bb_start[1]), 
                        max(union_bb[2], bb_start[2]), max(union_bb[3], bb_start[3]))
            end_pt = points[-1]
            self._fill_disk_poly_g(self.buffer_g, end_pt[0], end_pt[1], radius, color)
            bb_end = self._bbox_disk(end_pt[0], end_pt[1], radius)
            union_bb = (min(union_bb[0], bb_end[0]), min(union_bb[1], bb_end[1]), 
                        max(union_bb[2], bb_end[2]), max(union_bb[3], bb_end[3]))
        return union_bb

    def _generate_dynamic_disk_poly(self, cx, cy, radius, seg_len_px=3.5, max_steps=720, min_steps=12):
        if radius <= 0: return []
        circumference = 2 * math.pi * abs(radius)
        steps = int(circumference / seg_len_px)
        steps = max(min_steps, min(steps, max_steps))
        poly = []
        angle_step = 2 * math.pi / steps
        for i in range(steps):
            angle = i * angle_step
            poly.append([cx + radius * math.cos(angle), cy + radius * math.sin(angle)])
        return poly    

    def _fill_disk_poly_g(self, g, cx, cy, r, c):
        if r > 0:
            poly = self._generate_dynamic_disk_poly(cx, cy, r)
            if poly: self._fill_polygon_aet_fxp(poly, c, g=g)

    def _bbox_disk(self,cx,cy,r): return(cx-r,cy-r,cx+r,cy+r)

    def _ensure_buffer(self,c):
        hp.dimgrob(self.buffer_g, self.W, self.H, c)

    def _draw_turtle_cursor(self):
        if not self.is_visible or self.viewport: return

        poly_outer = self._get_turtle_poly_phys()
        n = len(poly_outer)
        
        sum_x = sum(p[0] for p in poly_outer)
        sum_y = sum(p[1] for p in poly_outer)
        center_x = sum_x / n
        center_y = sum_y / n
        
        scale = 0.6
        poly_inner = []
        for px, py in poly_outer:
            nx = center_x + (px - center_x) * scale
            ny = center_y + (py - center_y) * scale
            poly_inner.append([nx, ny])

        black = 0x000000
        for i in range(n):
            p1 = poly_outer[i]
            p2 = poly_outer[(i+1)%n]
            hp.line(self.screen_g, int(p1[0]), int(p1[1]), int(p2[0]), int(p2[1]), black)

        white = 0xFFFFFF
        for i in range(n):
            p1 = poly_inner[i]
            p2 = poly_inner[(i+1)%n]
            hp.line(self.screen_g, int(p1[0]), int(p1[1]), int(p2[0]), int(p2[1]), white)

    def _present_full(self):
        if self.viewport: return 
        if self.buffer_g != self.screen_g: 
            hp.blit(self.screen_g, 0, 0, self.buffer_g)
        self._draw_turtle_cursor()
    
    def _mark_dirty(self,x0,y0,x1,y1,m=1):
        if self.viewport: return
        x0,x1=min(x0,x1),max(x0,x1); y0,y1=min(y0,y1),max(y0,y1)
        x0=int(x0)-m; y0=int(y0)-m; x1=int(x1)+m; y1=int(y1)+m
        if self._dirty is None: self._dirty=(x0,y0,x1,y1)
        else:
            dx0,dy0,dx1,dy1=self._dirty
            self._dirty=(min(dx0,x0),min(dy0,y0),max(dx1,x1),max(dy1,y1))
            
    def _present_dirty(self):
        if self.viewport: return
        
        if self._tracer_step == 0: return 
        if self._tracer_step > 1:
            self._tracer_counter += 1
            if self._tracer_counter < self._tracer_step: return
            self._tracer_counter = 0

        if self.buffer_g != self.screen_g and self._dirty:
            x0, y0, x1, y1 = self._dirty
            x0 = max(0, x0); y0 = max(0, y0)
            x1 = min(self.W - 1, x1); y1 = min(self.H - 1, y1)
            
            if x1 >= x0 and y1 >= y0:
                hp.strblit2(self.screen_g, x0, y0, x1 - x0 + 1, y1 - y0 + 1,
                            self.buffer_g, x0, y0, x1 - x0 + 1, y1 - y0 + 1)
            self._dirty = None

        self._draw_turtle_cursor()

        if self.delay_ms > 0:
            current_ticks = hp.ticks()
            elapsed_ms = current_ticks - self._last_frame_ticks
            wait_duration = self.delay_ms - elapsed_ms
            if wait_duration > 0: self.wait(wait_duration)
        
        self._last_frame_ticks = hp.ticks()

    def _bbox_of_points(self,pts):
        if not pts: return(0,0,0,0)
        xs=[p[0]for p in pts]; ys=[p[1]for p in pts]
        return(min(xs),min(ys),max(xs),max(ys))

    def _quad_for_thick_line(self, x1, y1, x2, y2, thickness):
        dx, dy = x2 - x1, y2 - y1
        L2 = dx*dx + dy*dy
        if L2 == 0:
            h = thickness / 2.0
            return [[x1-h,y1-h],[x1+h,y1-h],[x1+h,y1+h],[x1-h,y1+h]]
        invL = 1.0 / (L2**0.5)
        nx, ny = -dy * invL, dx * invL
        h = thickness / 2.0
        hx, hy = nx * h, ny * h
        return [[x1+hx,y1+hy],[x2+hx,y2+hy],[x2-hx,y2-hy],[x1-hx,y1-hy]]

    def _generate_capsule_polygon(self, p1, p2, thickness):
        radius = thickness / 2.0
        if radius <= 0: return []
        x1, y1 = p1; x2, y2 = p2
        dx, dy = x2 - x1, y2 - y1
        if dx == 0 and dy == 0: return self._generate_dynamic_disk_poly(x1, y1, radius)
        if thickness < 2.5: return self._quad_for_thick_line(x1, y1, x2, y2, thickness)

        line_angle = math.atan2(dy, dx)
        poly = []
        for i in range(13): 
            angle = line_angle - math.pi/2 + (i * math.pi / 12)
            poly.append([x2 + radius * math.cos(angle), y2 + radius * math.sin(angle)])
        for i in range(13): 
            angle = line_angle + math.pi/2 + (i * math.pi / 12)
            poly.append([x1 + radius * math.cos(angle), y1 + radius * math.sin(angle)])
        return poly

    def _render_thick_polyline(self, points, thickness, color, closed=False):
        if self.viewport:
            norm_pts = self._p2n_points(points)
            self._display_list.append(('polyline', norm_pts, thickness, color, closed))
            return (0,0,0,0)
        
        t = int(round(thickness))
        n = len(points)
        if t <= 1:
            if n > 1:
                bbox = self._bbox_of_points(points)
                num_segs = n if closed else n-1
                for i in range(num_segs):
                    p1,p2 = points[i],points[(i+1)%n]
                    hp.line(self.buffer_g,int(p1[0]+0.5),int(p1[1]+0.5),int(p2[0]+0.5),int(p2[1]+0.5),color)
                return bbox
            return (0,0,0,0)
        
        if n == 2 and not closed:
            poly = self._generate_capsule_polygon(points[0], points[1], t)
            self._fill_polygon_aet_fxp(poly, color, g=self.buffer_g)
            return self._bbox_of_points(poly)

        elif n > 1:
            union_bb = None
            def update_union_bb(bb):
                nonlocal union_bb
                if not bb: return
                if union_bb is None: union_bb = list(bb)
                else:
                    union_bb[0]=min(union_bb[0],bb[0]); union_bb[1]=min(union_bb[1],bb[1])
                    union_bb[2]=max(union_bb[2],bb[2]); union_bb[3]=max(union_bb[3],bb[3])

            num_segs = n if closed else n-1
            for i in range(num_segs):
                p1, p2 = points[i], points[(i+1)%n]
                quad = self._quad_for_thick_line(p1[0], p1[1], p2[0], p2[1], t)
                self._fill_polygon_aet_fxp(quad, color, g=self.buffer_g)
                update_union_bb(self._bbox_of_points(quad))
            
            radius = t / 2.0
            num_verts = n if closed else n
            for i in range(num_verts):
                cx, cy = points[i]
                self._fill_disk_poly_g(self.buffer_g, cx, cy, radius, color)
                update_union_bb(self._bbox_disk(cx, cy, radius))
            
            return union_bb
        return (0,0,0,0)

    def _fill_polygon_aet_fxp(self, points_flt, color, g=0, clamp_to_graphic=True):
        if not points_flt or len(points_flt) < 3: return
        
        if self.viewport:
            norm_pts = self._p2n_points(points_flt)
            self._display_list.append(('polygon', norm_pts, color))
            return

        pts = [[int(p[0] * self._FIXED_POINT_SCALE), int(round(p[1]))] for p in points_flt]
        min_y, max_y = min(p[1] for p in pts), max(p[1] for p in pts)
        
        scan_min, scan_max = min_y, max_y
        if clamp_to_graphic:
            H = self.H
            if max_y < 0 or min_y >= H: return
            scan_min, scan_max = max(min_y, 0), min(max_y, H - 1)
            
        y0 = min_y
        edge_table = [[] for _ in range(max_y - min_y + 1)]
        for i in range(len(pts)):
            x1, y1 = pts[i]
            x2, y2 = pts[(i+1)%len(pts)]
            if y1 == y2: continue
            if y1 > y2: x1,x2,y1,y2 = x2,x1,y2,y1
            edge_table[y1 - y0].append([y2, x1, (x2-x1)//(y2-y1)])
            
        active = []
        for y in range(min_y, scan_min):
            idx = y - y0
            if 0 <= idx < len(edge_table): active.extend(edge_table[idx])
            active = [e for e in active if e[0] > y]
            for e in active: e[1] += e[2]

        for y in range(scan_min, scan_max + 1):
            idx = y - y0
            if 0 <= idx < len(edge_table): active.extend(edge_table[idx])
            active = [e for e in active if e[0] > y]
            active.sort(key=lambda e: e[1])
            for i in range(0, len(active) - 1, 2):
                x_start = (active[i][1] + self._FIXED_POINT_CEIL_ADD) >> self._FIXED_POINT_SHIFT
                x_end = (active[i+1][1] - 1) >> self._FIXED_POINT_SHIFT
                if x_start <= x_end: hp.line(g, x_start, y, x_end, y, color)
            for e in active: e[1] += e[2]

    # def _fill_polygon_aet_fxp(self, points_flt, color, g=0, clamp_to_graphic=True):
    #     if not points_flt or len(points_flt) < 3: return
    #     if self.viewport:
    #         norm_pts = self._p2n_points(points_flt)
    #         self._display_list.append(('polygon', norm_pts, color))
    #         return
    #     pts_int = [[int(p[0]), int(p[1])] for p in points_flt]
    #     pts_str = str(pts_int)
    #     cmd = "FILLPOLY_P(G{},{},{})".format(int(g), pts_str, int(color))
    #     self._ppl_eval_safe(cmd)

    def _int2rgb(self, c):
        if c is None: return None
        r = (c >> 16) & 0xFF
        g = (c >> 8) & 0xFF
        b = c & 0xFF
        if self._colormode == 1.0:
            return (r / 255.0, g / 255.0, b / 255.0)
        else:
            return (r, g, b)

    def _parse_color(self, c):
        def clamp8(v): return max(0, min(255, int(round(v))))
        if isinstance(c, int): return int(c) & 0xFFFFFF
        if isinstance(c, str):
            s = c.strip().lower()
            if s in self._COLOR_NAMES: r,g,b = self._COLOR_NAMES[s]
            elif s.startswith('#'):
                h=s[1:];l=len(h)
                if l==3:r,g,b=(int(v*2,16)for v in h)
                elif l==6:r,g,b=(int(h[i:i+2],16)for i in(0,2,4))
                else: raise ValueError("Bad hex color")
            else: raise ValueError("Unknown color string")
            return (r<<16)|(g<<8)|b
        if isinstance(c,(tuple,list)):
            if len(c)!=3:raise ValueError("Color tuple must be (r,g,b)")
            cm=self._colormode
            try: is_float=any(isinstance(v,float)for v in c)or max(v for v in c)<=1.0
            except: is_float=True
            if is_float:r,g,b=(clamp8(v*255.0)for v in c)
            else:r,g,b=(clamp8((v/cm)*255.0)for v in c)
            return (r<<16)|(g<<8)|b
        raise ValueError("Unsupported color type")

    def _ppl_eval_safe(self, cmd):
        old_sep = 0
        try: old_sep = int(hp.eval("HSeparator"))
        except: pass
        result = None
        try:
            hp.eval("HSeparator:=0")
            result = hp.eval(cmd)
        except Exception: pass
        finally:
            try: hp.eval("HSeparator:={}".format(old_sep))
            except: pass
        return result

    def _get_text_size_ppl(self, text, font):
        safe_text = str(text).replace('"', '""')
        cmd = 'TEXTSIZE("{}", {})'.format(safe_text, int(font))
        res = self._ppl_eval_safe(cmd)
        if res is None: return 0, 0
        try:
            if isinstance(res, str):
                clean = res.strip("{}")
                parts = clean.split(',')
                return int(parts[0]), int(parts[1])
            return int(res[0]), int(res[1])
        except: return 0, 0

    def _draw_text_ppl(self, text, g, x, y, font, color):
        safe_text = str(text).replace('"', '""')
        c = self._parse_color(color)
        cmd = 'TEXTOUT_P("{}", G{}, {}, {}, {}, {})'.format(
            safe_text, int(g), int(x), int(y), int(font), int(c)
        )
        self._ppl_eval_safe(cmd)
